package com.aps.service.Algorithm;

import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Algorithm.ObjectiveWeights;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * 增强版NSGA-II工具类（自定义权重+并行非支配排序）
 */
public class NSGAIIUtils {
    private ObjectiveWeights objectiveWeights;
    private int taskCount; // 并行任务数（默认=CPU核心数）

    // true 越小越好
    private boolean[] isMinimize = {true, true, true, true, true};

    public void init(ObjectiveWeights weights, int taskCount,boolean[] isMinimize) {
        this.objectiveWeights = (weights != null) ? weights : new ObjectiveWeights();
        this.taskCount = (taskCount > 0) ? taskCount : Runtime.getRuntime().availableProcessors()-1;
        this.isMinimize=isMinimize;
    }

    /**
     * 初始化（设置权重和并行任务数）
     */
    public void init(ObjectiveWeights weights, int taskCount) {
        this.objectiveWeights = (weights != null) ? weights : new ObjectiveWeights();
        this.taskCount = (taskCount > 0) ? taskCount : Runtime.getRuntime().availableProcessors()-1;
    }

    /**
     * 重载初始化方法（默认任务数）
     */
    public void init(ObjectiveWeights weights) {
        init(weights, 0);
    }


    /**
     * 并行快速非支配排序（大规模种群提速50%+）
     */
    public List<List<Chromosome>> parallelFastNonDominatedSort(List<Chromosome> population) {
        int popSize = population.size();
        List<List<Chromosome>> fronts = new ArrayList<>();

        // 步骤1：预处理 - 计算归一化目标值和加权目标值
        preprocessObjectives(population);
        if (popSize <= taskCount * 15) { // 小规模种群直接串行 7*15
            fronts = fastNonDominatedSort(population);
        } else {

            int chunkCount = Math.min(taskCount, popSize);
            // 步骤2：拆分种群为多个块，并行计算支配关系
            List<List<Chromosome>> chunks = splitPopulation(population, chunkCount);

            Map<Chromosome, List<Chromosome>> dominanceMatrix = new HashMap<>();
            Map<Chromosome, Integer> dominatedCount = new HashMap<>();
            for (Chromosome chromo : population) {
                dominanceMatrix.put(chromo, new ArrayList<>());
                dominatedCount.put(chromo, 0);
            }


            ExecutorService executor = Executors.newFixedThreadPool(chunkCount);
            try {
                List<Future<?>> futures = new ArrayList<>(); // 存储任务Future，确保全量等待
                for (List<Chromosome> chunk : chunks) {
                    futures.add(executor.submit(() -> {
                        for (Chromosome p : chunk) {
                            for (Chromosome q : population) {
                                if (p == q) continue;
                                if (dominates(p, q)) {
                                    synchronized (dominanceMatrix.get(p)) { // 细粒度锁
                                        dominanceMatrix.get(p).add(q);
                                    }
                                } else if (dominates(q, p)) {
                                    synchronized (dominatedCount) {
                                        dominatedCount.put(p, dominatedCount.get(p) + 1);
                                    }
                                }
                            }
                        }
                    }));
                }

                // 核心修复2：等待所有任务完成，避免提前终止导致的计算不完整
                for (Future<?> future : futures) {
                    try {
                        future.get(); // 等待单个任务完成，抛出异常则终止
                    } catch (ExecutionException e) {
                        throw new RuntimeException("并行计算支配关系失败", e.getCause());
                    }
                }

                executor.shutdown();

            }catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("并行计算支配关系中断", e);
            } finally {
                if (!executor.isShutdown()) {
                    executor.shutdownNow(); // 确保线程池关闭
                }
            }



            // 生成前沿
            //  染色体 -> 数组下标的映射表
            Map<Chromosome, Integer> chromosomeIndexMap = new HashMap<>(popSize);
            int[] dominatedCounts = new int[popSize];
// 新增：支配矩阵数组化 - 每个p的下标，对应一个存储q下标的List
            List<List<Integer>> dominanceMatrixArr = new ArrayList<>(popSize);
            // ======== 修复空指针的核心行：循环popSize次，给每个索引塞一个空的ArrayList ========
            for (int idx = 0; idx < popSize; idx++) {
                dominanceMatrixArr.add(new ArrayList<>()); // 必须加这行，初始化每个位置的空列表
                Chromosome c = population.get(idx);
                chromosomeIndexMap.put(c, idx);
            }
            // 5. 填充映射表、被支配数数组、支配矩阵数组
            for (int idx = 0; idx < popSize; idx++) {
                Chromosome c = population.get(idx);

                dominatedCounts[idx] = dominatedCount.get(c);
                // 把HashMap的支配列表，转成q的下标列表，存入数组
                List<Chromosome> qList = dominanceMatrix.get(c);
                List<Integer> qIdxList = dominanceMatrixArr.get(idx); // 此时get(idx)绝对非空！
                for (Chromosome q : qList) {
                    Integer qIdx = chromosomeIndexMap.get(q);
                    if(qIdx != null){ // 额外兜底：防止q是null导致的空指针
                        qIdxList.add(qIdx);
                    }
                }
            }
            // ========== 初始化前沿列表，必须先添加F1（第一个前沿） ==========

            List<Chromosome> firstFront = new ArrayList<>();
// 遍历种群，找到所有被支配数=0的个体 → 这就是F1，绝对不被任何个体支配的最优解
            for (Chromosome chromosome : population) {
                if (chromosome == null) continue;
                int count = dominatedCount.getOrDefault(chromosome, 0);
                if (count == 0) {
                    firstFront.add(chromosome);
                }
            }
            fronts.add(firstFront);



            boolean[] addedFlag = new boolean[popSize];
            int i = 0;
            while (i < fronts.size() && !fronts.get(i).isEmpty()) {
                List<Chromosome> currentFront = fronts.get(i);
                List<Chromosome> nextFront = new ArrayList<>();
                Arrays.fill(addedFlag, false);

                for (Chromosome p : currentFront) {
                    if (p == null) continue;
                    Integer pIdxObj = chromosomeIndexMap.get(p);
                    if (pIdxObj == null || pIdxObj >= popSize || pIdxObj < 0) continue;
                    int pIdx = pIdxObj;

                    // 拿到p能支配的所有q的下标，无空指针
                    List<Integer> qIdxList = dominanceMatrixArr.get(pIdx);
                    for (int qIdx : qIdxList) {
                        // 核心逻辑：被支配数-1
                        if (dominatedCounts[qIdx] <= 0) {
                           continue;
                        }
                        dominatedCounts[qIdx]--;
                        // 当被支配数=0时，加入下一个前沿，去重
                        if (dominatedCounts[qIdx] == 0 && !addedFlag[qIdx]) {
                            addedFlag[qIdx] = true;
                            nextFront.add(population.get(qIdx));
                        }
                    }
                }
                // 把下一个前沿加入列表，继续循环
                if (!nextFront.isEmpty()) {
                    fronts.add(nextFront);
                }
                i++;
            }
        }
        assignRankToSolutions(fronts);
        return fronts;
    }

    public List<List<Chromosome>> parallelFastNonDominatedSort1(List<Chromosome> population) {
        int popSize = population.size();
        List<List<Chromosome>> fronts = new ArrayList<>();

        // 步骤1：预处理 - 计算归一化目标值和加权目标值
        preprocessObjectives(population);
        if (popSize <= taskCount * 15) { // 小规模种群直接串行 7*15
            fronts = fastNonDominatedSort(population);
        } else {



            int chunkCount = Math.min(taskCount, popSize);
            // 步骤2：拆分种群为多个块，并行计算支配关系
            List<List<Chromosome>> chunks = splitPopulation(population, chunkCount);
              ConcurrentHashMap<Chromosome, ConcurrentLinkedQueue<Chromosome>> dominanceMatrix = new ConcurrentHashMap<>();
              ConcurrentHashMap<Chromosome, AtomicInteger> dominatedCount = new ConcurrentHashMap<>();
            // 初始化字典
            for (Chromosome chromo : population) {
                dominanceMatrix.putIfAbsent(chromo, new ConcurrentLinkedQueue<>());
                dominatedCount.putIfAbsent(chromo, new AtomicInteger(0));
            }


            // 并行计算支配关系
            chunks.parallelStream().forEach(chunk -> {
                for (Chromosome p : chunk) {
                    for (Chromosome q : population) {
                        if (p == q) continue;

                        if (dominates(p, q)) {
                            dominanceMatrix.get(p).add(q);
                        } else if (dominates(q, p)) {
                            dominatedCount.get(p).incrementAndGet();
                        }
                    }
                }
            });



            // 步骤3：生成前沿

            List<Chromosome> front1 = new ArrayList<>();

            for (Chromosome p : population) {
                if (dominatedCount.get(p).get() == 0) {

                    front1.add(p);
                }
            }

            fronts.add(front1);
            int i = 0;

            while (!fronts.get(i).isEmpty()) {
                List<Chromosome> nextFront = new ArrayList<>();
                for (Chromosome p : fronts.get(i)) {
                    for (Chromosome q : dominanceMatrix.get(p)) {
                        AtomicInteger count = dominatedCount.get(q);
                        if (count.decrementAndGet() == 0) {

                            nextFront.add(q);
                        }
                    }
                }
                i++;

                fronts.add(nextFront);

            }

            List<List<Chromosome>> fronts1=  fastNonDominatedSort(population);

        }
        assignRankToSolutions(fronts);
        return fronts;
    }


    /**
     * 用Stream API为分层后的解批量赋值Rank
     */
    public  void assignRankToSolutions(List<List<Chromosome>> result) {
        if (result == null || result.isEmpty()) {
            return;
        }

        // 遍历外层List，同时获取索引和对应的解列表
        IntStream.range(0, result.size()).forEach(i -> {
            int rank = i + 1;
            List<Chromosome> solutionsInRank = result.get(i);

            if(solutionsInRank!=null&&solutionsInRank.size()>0)
            {
                solutionsInRank.stream()
                    .filter(Objects::nonNull) // 过滤null的Solution
                    .forEach(solution -> solution.setRank(rank));

                if(solutionsInRank.size()>1) {
                    parallelCalculateCrowdingDistance(solutionsInRank);
                }
            }
        });
    }


    /**
     * 串行快速非支配排序（兼容小规模种群）
     */
    public List<List<Chromosome>> fastNonDominatedSort(List<Chromosome> population) {
        // 1. 初始化数据结构：S（被支配集合）、n（支配数）
        //当前解能支配的其他解的集合。
        Map<Chromosome, List<Chromosome>> S = new HashMap<>();
        //能支配当前解的其他解的数量；
        Map<Chromosome, Integer> n = new HashMap<>();
        List<List<Chromosome>> fronts = new ArrayList<>();

        // 为每个个体预先初始化 S 和 n
        for (Chromosome p : population) {
            S.put(p, new ArrayList<>()); // 初始化空集合，避免null 当前解能支配的其他解的集合
            n.put(p, 0); // 支配数初始化为0 。能支配当前解的其他解的数量
        }
        //S 当前解能支配的其他解的集合。
        //n 能支配当前解的其他解的数量；
        // 2. 计算每个个体的支配关系
        for (Chromosome s1 : population) {
            for (Chromosome s2 : population) {
                if (s1 == s2) continue;
                // p.getWeightedObjective() < q.getWeightedObjective();
                //【0,1】 越靠近1 越好

                if (dominates(s1, s2)) {
                    //s1支配s2 → 更新s1的被支配集合，s2的支配数+1
                    S.get(s1).add(s2);
                } else if (dominates(s2, s1)) {
                    // s2支配s1 → s1的被支配数+1（避免重复计算）
                    n.put(s1, n.get(s1) + 1);
                }
            }

            // 3. 初始化第一前沿（支配数为0的个体）
            if (n.get(s1) == 0) {
                if (fronts.isEmpty()) {
                    fronts.add(new ArrayList<>());
                }
                // s1.setRank(1);
                fronts.get(0).add(s1);
            }
        }

        // 4. 生成后续前沿
        int i = 0;
        while (!fronts.get(i).isEmpty()) {
            List<Chromosome> nextFront = new ArrayList<>();
            for (Chromosome p : fronts.get(i)) {
                // 遍历p的被支配集合S.get(p)（此时已初始化，不会为null）
                for (Chromosome q : S.get(p)) {
                    n.put(q, n.get(q) - 1);
                    if (n.get(q) == 0) {

                        nextFront.add(q);
                    }
                }
            }
            i++;
            fronts.add(nextFront);
        }

        return fronts;
    }

    /**
     * 预处理目标值（归一化+加权）
     */
    private void preprocessObjectives(List<Chromosome> population) {

        int kpisize = population.get(0).getObjectives().length;
        // 计算每个目标的最大/最小值
        double[] minValues = new double[kpisize];
        double[] maxValues = new double[kpisize];
//        for (int i = 0; i < kpisize; i++) {
//            final int idx = i;
//            minValues[i] = population.stream().mapToDouble(c -> c.getObjectives()[idx]).min().orElse(0);
//            maxValues[i] = population.stream().mapToDouble(c -> c.getObjectives()[idx]).max().orElse(0);
//        }
        for (int i = 0; i < kpisize; i++) {
            minValues[i] = Double.MAX_VALUE;
            maxValues[i] = Double.MIN_VALUE;
        }
        for (Chromosome chromo : population) {
            double[] obj = chromo.getObjectives();
            for (int i = 0; i < kpisize; i++) {
                minValues[i] = Math.min(minValues[i], obj[i]);
                maxValues[i] = Math.max(maxValues[i], obj[i]);
            }
        }

        // 并行计算归一化和加权目标值
        population.parallelStream().forEach(chromo -> {

            double[] normalized = objectiveWeights.normalizeObjectives(chromo.getObjectives(), minValues, maxValues,this.isMinimize);
            chromo.setMaxObjectives(maxValues);
            chromo.setMinObjectives(minValues);
            chromo.setWeightedObjectives(normalized);
            if (objectiveWeights.isPureNSGAIIMode()) {
                chromo.setWeightedObjective(objectiveWeights.calculateObjective(normalized));
            } else {
                chromo.setWeightedObjective(objectiveWeights.calculateWeightedObjective(normalized));
            }
        });
    }

    /**
     * 拆分种群为多个块（用于并行计算）
     */
    private List<List<Chromosome>> splitPopulation(List<Chromosome> population, int chunkCount) {
        List<List<Chromosome>> chunks = new ArrayList<>();
        int chunkSize = (int) Math.ceil((double) population.size() / chunkCount);

        for (int i = 0; i < population.size(); i += chunkSize) {
            int end = Math.min(i + chunkSize, population.size());
            chunks.add(population.subList(i, end));
        }

        return chunks;
    }


    /**
     * 判断p是否支配q（支持自定义权重）
     */
    public boolean dominates(Chromosome p, Chromosome q) {
        // 纯NSGA-II模式：标准帕累托支配
        double[] pObjectives = p.getWeightedObjectives();
        double[] qObjectives = q.getWeightedObjectives();
        if (objectiveWeights.isPureNSGAIIMode()) {
            boolean hasBetter = false; // 是否至少有一个目标更优
            boolean hasWorse = false;  // 是否有任何一个目标更差


            for (int i = 0; i < pObjectives.length; i++) {

                double thisObj = pObjectives[i];
                double otherObj = qObjectives[i];

                // 目标越大越好：当前解 > 另一个解 → 更优
                if (thisObj > otherObj) {
                    hasBetter = true;
                }
                // 当前解 < 另一个解 → 更差
                else if (thisObj < otherObj) {
                    hasWorse = true;
                }
            }

            return hasBetter && !hasWorse;
        }
        // 加权NSGA-II模式：融合权重的支配关系
        else {
            // 方式1：加权目标值更大则支配（简单高效）
            //WeightedObjective 已计算，归一法【0,1】,越靠近1 越好
            double thisScore = p.getWeightedObjective();
            double otherScore = q.getWeightedObjective();
            // - 加权得分严格更优；
            // - 至少一个权重>0的目标上不劣于对方（避免极端情况）
            boolean hasBetterScore = thisScore > otherScore;
            boolean hasAtLeastOneNotWorse = false;
            for (int i = 0; i < p.getObjectives().length; i++) {
                if (objectiveWeights.getWeights()[i] <= 0) continue; // 跳过权重为0的目标
                double thisObj = pObjectives[i];
                double otherObj = qObjectives[i];
                if (thisObj >= otherObj) {
                    hasAtLeastOneNotWorse = true;
                    break;
                }
            }
            return hasBetterScore && hasAtLeastOneNotWorse;
        }
    }

    /**
     * 并行计算拥挤距离
     */
    public  void parallelCalculateCrowdingDistance(List<Chromosome> front) {
        if (front.isEmpty()) return;

        int kpisize = front.get(0).getObjectives().length;
        int popSize = front.size();

        // 初始化距离为0
       // front.parallelStream().forEach(chromo -> chromo.setCrowdingDistance(0.0));


        // 计算每个目标的最大/最小值
        double[] idealPoint = new double[kpisize];

        for (int i = 0; i < kpisize; i++) {
            final int idx = i;
            if(isMinimize[i])
            {
                idealPoint[i] = front.stream().mapToDouble(c -> c.getObjectives()[idx]).min().orElse(0);

            }else {
                idealPoint[i] = front.stream().mapToDouble(c -> c.getObjectives()[idx]).max().orElse(0);

            }
        }

        if(popSize>taskCount*15)
        {
            // 对每个目标维度并行计算距离
            IntStream.range(0, popSize).parallel().forEach(m -> {
                // 按当前目标排序
                Chromosome chromo=front.get(m);
                double[] idealPoint1 = idealPoint;
                double[] objectives = chromo.getObjectives();
                double distance = calculateEuclideanDistance(objectives, idealPoint1);

                chromo.setCrowdingDistance(distance);
            });
        }else {
            front.forEach(chromo -> {

                double[] idealPoint1 = idealPoint;
                double[] objectives = chromo.getObjectives();
                double distance = calculateEuclideanDistance(objectives, idealPoint1);

                chromo.setCrowdingDistance(distance);
            });
        }


    }

    /**
     * 计算目标值列表与理想点的欧氏距离
     * @param objectives 解的目标值列表
     * @param idealPoint 理想点
     * @return 欧氏距离（值越小越优）
     */
    private  double calculateEuclideanDistance(double[] objectives, double[] idealPoint) {
        double sum = 0.0;
        for (int i = 0; i < objectives.length; i++) {
            double diff = objectives[i] - idealPoint[i]; //与理想值的差
            sum += diff * diff; // 平方和
        }
        return Math.sqrt(sum); // 开平方得到欧氏距离
    }

    /**
     * 选择下一代种群（兼容并行计算）
     */
    public List<Chromosome> selectNextPopulation(List<List<Chromosome>> fronts, int populationSize) {
        List<Chromosome> nextPopulation = new ArrayList<>();
        int currentSize = 0;

        for (List<Chromosome> front : fronts) {
            if (currentSize + front.size() <= populationSize) {
                nextPopulation.addAll(front);
                currentSize += front.size();
            } else {

                // 按拥挤距离排序，取前N个
                List<Chromosome> sortedByDistance = front.stream()
                        .sorted((c1, c2) -> Double.compare(c2.getCrowdingDistance(), c1.getCrowdingDistance()))
                        .collect(Collectors.toList());
                nextPopulation.addAll(sortedByDistance.subList(0, populationSize - currentSize));
                currentSize = populationSize;
                break;
            }
        }

        return nextPopulation;
    }

    public void Test() {

        ObjectiveWeights weights = new ObjectiveWeights();
        double[] weightvs = new double[] { 0.5, 0.5 };
        weights.setWeights(weightvs);
        weights.setPureNSGAIIMode(false);
        init(weights);
        List<Chromosome> chromosomeList = new ArrayList<>();
        Chromosome chromosome1 = new Chromosome();
        chromosome1.setID("A");
        chromosome1.setObjectives(new double[]{1, 3});
        chromosomeList.add(chromosome1);
        Chromosome chromosome4 = new Chromosome();
        chromosome4.setID("B");
        chromosome4.setObjectives(new double[]{2, 2});
        chromosomeList.add(chromosome4);
        Chromosome chromosome5 = new Chromosome();
        chromosome5.setID("C");
        chromosome5.setObjectives(new double[]{3, 1});
        chromosomeList.add(chromosome5);
        Chromosome chromosome2 = new Chromosome();
        chromosome2.setID("D");
        chromosome2.setObjectives(new double[]{2, 3});
        chromosomeList.add(chromosome2);
        Chromosome chromosome3 = new Chromosome();
        chromosome3.setID("E");
        chromosome3.setObjectives(new double[]{4, 4});
        chromosomeList.add(chromosome3);
        parallelFastNonDominatedSort(chromosomeList);

//        解 A（1,3）：
//        支配的解：D（2,3）、E（4,4）→ S (A) = {D,E}
//        被支配的解：无 → n (A) = 0
//        解 B（2,2）：
//        支配的解：D（2,3）、E（4,4）→ S (B) = {D,E}
//        被支配的解：无 → n (B) = 0
//        解 C（3,1）：
//        支配的解：E（4,4）→ S (C) = {E}
//        被支配的解：无 → n (C) = 0
//        解 D（2,3）：
//        支配的解：E（4,4）→ S (D) = {E}
//        被支配的解：A、B → n (D) = 2
//        解 E（4,4）：
//        支配的解：无 → S (E) = ∅
//        被支配的解：A、B、C、D → n (E) = 4

    }


}
