package com.aps.service.impl;

import com.aps.common.util.R;
import com.aps.common.util.SnowFlackIdWorker;
import com.aps.entity.*;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Algorithm.GAScheduleResult;
import com.aps.entity.basic.Order;
import com.aps.mapper.RoutingDetailMapper;
import com.aps.mapper.RoutingHeaderMapper;
import com.aps.service.*;
import com.aps.service.plan.SceneService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;
import java.util.stream.Collectors;

import static org.springframework.beans.BeanUtils.copyProperties;

/**
 • 设备日历服务实现类

 • 负责生成设备日历数据、处理订单数据转换等功能

 */
@Slf4j
@Service
@RequiredArgsConstructor
public class LanuchServiceImpl implements LanuchService {

    @Autowired
    ProdEquipSpecialCalService prodEquipSpecialCalService;
    @Autowired
    ApsOrderService apsOrderService;

    @Autowired
    RoutingHeaderService routingHeaderService;
    @Autowired
    ProdLaunchOrderService prodLaunchOrderService;
    @Autowired
    RoutingDetailMapper routingDetailMapper;
    @Autowired
    ProdProcessExecService prodProcessExecService;

    @Autowired
    RoutingDetailConnectService routingDetailConnectService;
    @Autowired
    ProdOrderProcessService prodOrderProcessService;

    @Autowired
    RoutingHeaderMapper routingHeaderMapper;
    @Autowired
    ProdSceneConfigService prodSceneConfigService;

    @Autowired
    ProdEquipmentService prodEquipmentService;
    @Autowired
    RoutingDetailEquipService routingDetailEquipService;
    @Autowired
    SceneService sceneService;
    @Autowired
    MesOrderService mesOrderService;
    @Autowired
    DispatchService dispatchService;
    @Autowired
    RoutingDiscreteParamService  routingDiscreteParamService;
    /**
     ◦ 生成场景数据

     *
     *
     ◦ @param sceneName        场景名称

     ◦ @param username         操作用户

     ◦ @return 处理结果

     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public R<String> lanuch( String sceneName, String username) {
        try {
            // 参数校验
            if (sceneName == null || sceneName.trim().isEmpty()) {
                return R.failed("场景名称不能为空");
            }



            // 1. 创建场景
            String sceneId = createScene(sceneName);
            if (sceneId == null) {
                return R.failed("场景名称已存在");
            }

            prodEquipSpecialCalService.copyFromEquipCapacityDef(sceneId);

            List<ApsOrder> apsOrders = apsOrderService.lambdaQuery()
                    .eq(ApsOrder::getIsdeleted, 0)
                    .like(ApsOrder::getCode, "佟礼测试数据")
                    .list();
            if (CollectionUtils.isEmpty(apsOrders)) {
                return R.failed("工单列表不能为空");
            }
            // 4. 处理订单数据转换
            List<ProdLaunchOrder> prodLaunchOrders = processOrderData(sceneId, apsOrders);


            // 5. 转换工单于工序执行表
            convertToProcessExecBatch(prodLaunchOrders, sceneId);


            // 6. 生成工序关联关系
            generateProcessRelationsBatch(prodLaunchOrders, sceneId);

            log.info("场景数据生成完成，场景ID：{}", sceneId);
            return R.ok("场景数据生成成功");
        } catch (Exception e) {
            log.error("生成场景数据失败", e);
            return R.failed("生成场景数据失败: " + e.getMessage());
        }
    }

    /**
     *复制数据
     * @param oldSceneName
     * @param newSceneName
     * @return R<String> Result
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public R<String> copyScene(String oldSceneName,String username, String newSceneName) {
        try {
            // Validate input parameters
            if (oldSceneName == null || oldSceneName.trim().isEmpty()) {
                return R.failed("场景ID不能为空");
            }

            if (newSceneName == null || newSceneName.trim().isEmpty()) {
                return R.failed("复制场景ID不能为空");
            }



            // 检查场景是否存在
            ProdSceneConfig oldScene = prodSceneConfigService.lambdaQuery()
                    .eq(ProdSceneConfig::getSceneName, oldSceneName)
                    .one();

            if (oldScene == null) {
                return R.failed("源场景不存在");
            }
            //  创建场景
            String newSceneId = createScene(newSceneName);
            if (newSceneId == null) {
                return R.failed("场景名称已存在");
            }

            // 复制数据
            copyProdLaunchOrders(oldScene.getSceneId(), newSceneId);
            copyProdProcessExecs(oldScene.getSceneId(), newSceneId);
            copyProdOrderProcesses(oldScene.getSceneId(), newSceneId);
            copyProdEquipments(oldScene.getSceneId(), newSceneId);
            copyProdEquipSpecialCals(oldScene.getSceneId(), newSceneId);
            
            // 复制染色体文件（排产结果）
            copyChromosomeFile(oldScene.getSceneId(), newSceneId);

            log.info("场景数据复制完成，从 {} 到 {}", oldSceneName, newSceneName);
            return R.ok("场景数据复制成功");

        } catch (Exception e) {
            log.error("复制场景数据失败", e);
            throw new RuntimeException("复制场景数据失败: " + e.getMessage(), e);
        }
    }



    /**
     *下发计划
     * @param sceneId
     * @return R<String> Result
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public R<String> exportPlan(String sceneId) {
        try {
            Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
            if (chromosome == null || chromosome.getResult() == null || chromosome.getResult().isEmpty()) {
                return R.failed("未找到排程结果");
            }

            List<Order> prodLanuchList = chromosome.getOrders();
            
            List<MesOrder> mesOrderList = new ArrayList<>();
            for (Order prodLaunchOrder : prodLanuchList) {
                MesOrder mesOrder = new MesOrder();

                mesOrder.setMesCode(prodLaunchOrder.getOrderId());
//                mesOrder.setSerie(prodLaunchOrder.getSerie());
                mesOrder.setMaterialId(prodLaunchOrder.getMaterialId());
                mesOrder.setQuantity((long) prodLaunchOrder.getQuantity());
//                mesOrder.setDemandStartDate(prodLaunchOrder.getStartDate());
//                mesOrder.setDemandFinishDate(prodLaunchOrder.getEndDate());
//                mesOrder.setRoutingId(prodLaunchOrder.getRoutingId() != null ? Long.valueOf(prodLaunchOrder.getRoutingId()) : null);
//                mesOrder.setProductCode(prodLaunchOrder.getMaterialCode());
//                mesOrder.setStatus(1L); // Default status
                mesOrderList.add(mesOrder);
            }
            

            boolean saved = mesOrderService.saveBatch(mesOrderList);
            
            if (saved) {
                log.info("插入 {} 条数据到mes_order", mesOrderList.size());
            } else {
                log.error("插入失败");
                return R.failed("插入mes_order失败");
            }

            List<ProdProcessExec> processExecList = prodProcessExecService.lambdaQuery()
                    .eq(ProdProcessExec::getSceneId, sceneId)
                    .list();



            
            // 遍历GAScheduleResult结果并转换为dispatch
            List<Dispatch> dispatches = new ArrayList<>();
            for (GAScheduleResult gaResult : chromosome.getResult()) {


                ProdProcessExec prodProcessExec = processExecList.stream()
                        .filter(processExec -> processExec.getExecId().equals(gaResult.getExecId()))
                        .findFirst()
                        .orElse(null);


                Dispatch dispatch = new Dispatch();
                // 更新相关字段
                dispatch.setENof(gaResult.getOrderId());
                dispatch.setQuantity(gaResult.getQuantity());

                LocalDateTime baseTime = chromosome.getBaseTime() != null ? chromosome.getBaseTime() : LocalDateTime.now();
                dispatch.setBeginTime(baseTime.plusSeconds(gaResult.getStartTime()));
                dispatch.setEndTime(baseTime.plusSeconds(gaResult.getEndTime()));
                dispatch.setENof(gaResult.getOrderId());
                // 设置状态等其他字段
                dispatch.setTaskSeq(prodProcessExec.getTaskSeq());
                dispatch.setMesCode(gaResult.getOrderId());
                dispatch.setRoutingDetailId(prodProcessExec.getRoutingDetailId());
                dispatch.setOpe(prodProcessExec.getRoutingDetailName());
                // 添加到列表中
                dispatches.add(dispatch);
            }
            
            // 批量保存到数据库
            boolean savedDispatch = dispatchService.saveOrUpdateBatch(dispatches);
            
            if (savedDispatch) {
                return R.ok("计划下发成功，共处理 " + dispatches.size() + " 条记录");
            } else {
                return R.failed("保存到数据库失败");
            }
        } catch (NumberFormatException e) {
            log.error("数据转换失败，场景ID: " + sceneId, e);
            return R.failed("数据转换失败，请检查数据格式: " + e.getMessage());
        } catch (Exception e) {
            log.error("下发计划失败，场景ID: " + sceneId, e);
            return R.failed("下发计划失败: " + e.getMessage());
        }
    }


    /**
     ◦ 处理订单数据：批量转换ProdOrderMain到ProdLaunchOrder

     *
     ◦ @param sceneId 场景ID

     */
    private List<ProdLaunchOrder> processOrderData(String sceneId, List<ApsOrder> prodOrderMainList) {


        // 批量转换并插入
        List<ProdLaunchOrder> launchOrderList = prodOrderMainList.stream()
                .map(order -> convertToLaunchOrder(order, sceneId))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());



        // 批量插入
        if (!launchOrderList .isEmpty()) {
            for (ProdLaunchOrder order : launchOrderList ) {
                try {
                    prodLaunchOrderService.save(order); // 单条插入
                } catch (Exception e) {
                    log.error("保存订单失败: {}", order.getOrderId(), e);
                }
            }
        }


        log.info("成功处理{}条订单数据", prodOrderMainList.size());

        return launchOrderList;
    }

    /**
     ◦ 将ProdOrderMain转换为ProdLaunchOrder

     *
     ◦ @param order    源订单

     ◦ @param sceneId  场景ID

     ◦ @return 转换后的ProdLaunchOrder

     */
    private ProdLaunchOrder convertToLaunchOrder(ApsOrder order, String sceneId) {
        try {
            ProdLaunchOrder launchOrder = new ProdLaunchOrder();
            launchOrder.setOrderId(order.getId());
            launchOrder.setSceneId(sceneId);
            launchOrder.setSerie(order.getSeries());
            launchOrder.setCreateUser(String.valueOf(order.getCreatoruserid()));
            launchOrder.setMaterialCode(order.getMmcode());
//            launchOrder.setStartDate(order.get);
            launchOrder.setEndDate(order.getDeliverytime());
//            launchOrder.setOrderPriority(order.getPrioritry());
            launchOrder.setOrderPriority(1);
            launchOrder.setQuantity(order.getQuantity());
            launchOrder.setMaterialId(order.getMmid());

            String mmid = order.getMmid();
            // 通过mmid查找对应的工艺
            if (mmid != null && !mmid.isEmpty()) {
                LambdaQueryWrapper<RoutingHeader> routingWrapper = new LambdaQueryWrapper<>();
                routingWrapper.eq(RoutingHeader::getMaterialId, mmid);
                routingWrapper.eq(RoutingHeader::getIsDeleted, 0); // 添加判断 is_deleted=0
                List<RoutingHeader> routingHeaders = routingHeaderService.list(routingWrapper);
                RoutingHeader routingHeader = null;
                if (!CollectionUtils.isEmpty(routingHeaders)) {
                    routingHeader = routingHeaders.get(0);
                }

                if (routingHeader != null) {
                    launchOrder.setRoutingId(routingHeader.getId());
                    launchOrder.setRoutingCode(defaultIfBlank(routingHeader.getCode(), ""));
                }
            }

            return launchOrder;
        } catch (Exception e) {
            log.error("转换订单[{}]失败", order.getId(), e);
            return null;
        }
    }

    /**
     ◦ 创建场景

     *
     ◦ @param sceneName 场景名称

     ◦ @return 场景ID，如果场景已存在则返回null

     */
    private String createScene(String sceneName) {
        // 检查场景名称是否已存在
        boolean exists = prodSceneConfigService.lambdaQuery()
                .eq(ProdSceneConfig::getSceneName, sceneName)
                .exists();

        if (exists) {
            log.warn("场景名称[{}]已存在", sceneName);
            return null;
        }

        // 创建新场景
        String sceneId = generateSceneId();
        ProdSceneConfig sceneConfig = new ProdSceneConfig();
        sceneConfig.setSceneId(sceneId);
        sceneConfig.setSceneName(sceneName);
        sceneConfig.setSceneStatus((short) 1);
        sceneConfig.setCreateTime(LocalDateTime.now());

        prodSceneConfigService.save(sceneConfig);
        log.info("创建新场景成功，场景ID：{}，名称：{}", sceneId, sceneName);

        return sceneId;
    }

    /**
     ◦ 生成场景ID

     *
     ◦ @return 格式化的场景ID

     */
    private String generateSceneId() {
        return UUID.randomUUID().toString().replace("-", "").toUpperCase();
    }

    /**
     ◦ 生成日历数据

     *
     ◦ @param sceneId           场景ID

     ◦ @param planResource      设备信息

     ◦ @param shiftWorkSchedList 班次信息

     ◦ @param startTime         开始时间

     ◦ @param endTime           结束时间

     */
    private void generateCalendarData(String sceneId, PlanResource planResource,
                                      List<MesShiftWorkSched> shiftWorkSchedList,
                                      LocalDateTime startTime, LocalDateTime endTime) {
        LocalDate currentDate = startTime.toLocalDate();
        LocalDate endDate = endTime.toLocalDate();

        // 批量保存列表
        List<ProdEquipSpecialCal> calendarList = new java.util.ArrayList<>();

        while (!currentDate.isAfter(endDate)) {
            int dayOfWeekValue = currentDate.getDayOfWeek().getValue();

            for (MesShiftWorkSched shiftWorkSched : shiftWorkSchedList) {
                if (shiftWorkSched.getStartWeekDay() == dayOfWeekValue) {
                    processSingleShift(sceneId, planResource, shiftWorkSched,
                            currentDate, startTime, endTime, calendarList);
                }
            }

            currentDate = currentDate.plusDays(1);
        }

        // 批量保存日历数据
        if (!CollectionUtils.isEmpty(calendarList)) {
            prodEquipSpecialCalService.saveBatch(calendarList);
            log.info("设备[{}]生成{}条日历记录", planResource.getReferenceCode(), calendarList.size());
        }
    }

    /**
     ◦ 处理单个班次

     *
     ◦ @param sceneId         场景ID

     ◦ @param planResource    设备信息

     ◦ @param shiftWorkSched  班次信息

     ◦ @param currentDate     当前日期

     ◦ @param startTime       开始时间

     ◦ @param endTime         结束时间

     ◦ @param calendarList    日历列表

     */
    private void processSingleShift(String sceneId, PlanResource planResource,
                                    MesShiftWorkSched shiftWorkSched, LocalDate currentDate,
                                    LocalDateTime startTime, LocalDateTime endTime,
                                    List<ProdEquipSpecialCal> calendarList) {
        LocalTime shiftStartTimeRaw = shiftWorkSched.getShiftStart().toLocalTime();
        LocalTime shiftEndTimeRaw = shiftWorkSched.getShiftEnd().toLocalTime();

        LocalDateTime shiftStartTime = LocalDateTime.of(currentDate, shiftStartTimeRaw);
        LocalDateTime shiftEndTime = LocalDateTime.of(currentDate, shiftEndTimeRaw);

        // 处理跨天班次
        if (shiftEndTimeRaw.isBefore(shiftStartTimeRaw)) {
            shiftEndTime = shiftEndTime.plusDays(1);
        }

        // 时间范围校验
        if (shiftStartTime.isAfter(endTime) || shiftEndTime.isBefore(startTime)) {
            return;
        }

        // 调整时间到范围内
        LocalDateTime actualStartTime = shiftStartTime.isBefore(startTime) ? startTime : shiftStartTime;
        LocalDateTime actualEndTime = shiftEndTime.isAfter(endTime) ? endTime : shiftEndTime;

        if (actualStartTime.isBefore(actualEndTime)) {
            ProdEquipSpecialCal equipSpecialCal = createCalendarRecord(sceneId, planResource,
                    shiftWorkSched, actualStartTime, actualEndTime);
            calendarList.add(equipSpecialCal);
        }
    }

    /**
     ◦ 创建日历记录

     *
     ◦ @param sceneId         场景ID

     ◦ @param planResource    设备信息

     ◦ @param shiftWorkSched  班次信息

     ◦ @param startTime       开始时间

     ◦ @param endTime         结束时间

     ◦ @return 日历记录

     */
    private ProdEquipSpecialCal createCalendarRecord(String sceneId, PlanResource planResource,
                                                     MesShiftWorkSched shiftWorkSched,
                                                     LocalDateTime startTime, LocalDateTime endTime) {
        ProdEquipSpecialCal equipSpecialCal = new ProdEquipSpecialCal();
        equipSpecialCal.setSceneId(sceneId);
        equipSpecialCal.setEquipId(planResource.getReferenceId());
        equipSpecialCal.setEquipCode(planResource.getReferenceCode());
        equipSpecialCal.setStartDate(startTime);
        equipSpecialCal.setEndDate(endTime);
        equipSpecialCal.setShiftWorkSchedCode(shiftWorkSched.getShiftWorkSchedCode());
        equipSpecialCal.setPeriodDesc(shiftWorkSched.getShiftName());
        equipSpecialCal.setId(String.valueOf(SnowFlackIdWorker.getId()));
        return equipSpecialCal;
    }

    /**
     ◦ 批量转换工单于工序执行表

     *
     ◦ @param order 工单列表

     ◦ @param sceneId          场景ID

     */
    private void convertToProcessExecBatch(List<ProdLaunchOrder> order, String sceneId) {
        if (CollectionUtils.isEmpty(order)) {
            log.warn("工单列表为空，跳过工序转换");
            return;
        }


        for (ProdLaunchOrder prodOrderMain : order) {
            try {
                convertToProcessExec(prodOrderMain, sceneId);
            } catch (Exception e) {
                log.error("转换工单[{}]到工序执行表失败", prodOrderMain.getOrderId(), e);
                continue;
            }
        }

        log.info("完成{}个工单的工序转换",order.size());
    }

    /**
     ◦ 转换工单到工序执行表

     *
     ◦ @param prodOrderMain  工单ID

     ◦ @param sceneId   场景ID

     ◦ @return 转换结果

     */
    public boolean convertToProcessExec(ProdLaunchOrder prodOrderMain, String sceneId) {


        // 查询对应的工艺
        RoutingHeader routingHeader = getRoutingHeader(prodOrderMain.getRoutingId());

        // 查询该工艺下的所有工序

        List<RoutingDetail> routingDetails = getRoutingDetails(routingHeader.getId());

        List<RoutingDetailEquip> routingDetailEquip = getRoutingDetailEquip(routingHeader.getId());


        // 批量插入新的工序工单记录
        batchInsertProcessExec(prodOrderMain, routingDetails, sceneId,routingDetailEquip);



        return true;
    }

    /**
     ◦ 获取工艺信息

     *
     ◦ @param routingId 工艺ID

     ◦ @return 工艺信息

     */
    private RoutingHeader getRoutingHeader(Integer routingId) {
        LambdaQueryWrapper<RoutingHeader> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RoutingHeader::getId, routingId)
                .eq(RoutingHeader::getIsDeleted, 0); // 添加 is_deleted=0 过滤条件
        RoutingHeader routingHeader = routingHeaderMapper.selectOne(wrapper);

        if (routingHeader == null) {
            log.error("未找到对应工艺: {}", routingId);
            throw new RuntimeException("未找到对应工艺: " + routingId);
        }

        return routingHeader;
    }

    /**
     ◦ 获取工艺工序列表

     *
     ◦ @param routingHeaderId 工艺头ID

     ◦ @return 工序列表

     */
    private List<RoutingDetail> getRoutingDetails(Integer routingHeaderId) {
        LambdaQueryWrapper<RoutingDetail> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RoutingDetail::getRoutingHeaderId, routingHeaderId)
                .eq(RoutingDetail::getIsDeleted, 0) // 添加 is_deleted=0 过滤条件
                .orderByAsc(RoutingDetail::getTaskSeq);

        List<RoutingDetail> routingDetails = routingDetailMapper.selectList(wrapper);

        if (CollectionUtils.isEmpty(routingDetails)) {
            log.error("工艺下无工序信息: {}", routingHeaderId);
            throw new RuntimeException("工艺下无工序信息: " + routingHeaderId);
        }

        return routingDetails;
    }


    /**
     ◦ 批量插入工序执行记录

     *
     ◦ @param prodOrderMain 工单信息

     ◦ @param routingDetails 工序列表

     ◦ @param sceneId       场景ID

     */
    private void batchInsertProcessExec(ProdLaunchOrder prodOrderMain,
                                        List<RoutingDetail> routingDetails,
                                        String sceneId, List<RoutingDetailEquip> routingDetailEquip)  {


        List<ProdProcessExec> processExecList = routingDetails.stream()
                .map(detail -> createProcessExec(prodOrderMain, detail, sceneId))
                .collect(Collectors.toList());

        batchInsertEquipMent(routingDetailEquip, sceneId,processExecList);

        // 批量插入
        if (!CollectionUtils.isEmpty(processExecList)) {
            prodProcessExecService.saveBatch(processExecList); // 假设MyBatis-Plus支持批量插入
        }
    }

    // 添加一个方法来更新execId映射
    /*
    private void updateExecIdMapping(List<ProdProcessExec> processExecList) {
        // 这里可以将processExecList中的数据添加到一个全局映射中，供其他方法使用
        // 由于当前设计限制，暂不实现全局映射，但保留此方法以备将来扩展
    }
    */

    /**
     ◦ 创建工序执行记录

     *
     ◦ @param prodOrderMain 工单信息

     ◦ @param detail        工序详情

     ◦ @param sceneId       场景ID

     ◦ @return 工序执行记录

     */
    private ProdProcessExec createProcessExec(ProdLaunchOrder prodOrderMain,
                                              RoutingDetail detail,
                                              String sceneId) {
        ProdProcessExec prodProcessExec = new ProdProcessExec();
        prodProcessExec.setExecId(UUID.randomUUID().toString().replace("-", ""));
        prodProcessExec.setOrderId(prodOrderMain.getOrderId());
        prodProcessExec.setRoutingDetailId(detail.getId());
        prodProcessExec.setTaskSeq(detail.getTaskSeq());
        prodProcessExec.setRoutingDetailName(detail.getName());
        prodProcessExec.setMachineId(detail.getEquipTypeId());
//        prodProcessExec.setResourceGroup(String.valueOf(detail.getResourceId()));
        prodProcessExec.setPlanQty(prodOrderMain.getQuantity());
        prodProcessExec.setTargetStartDate(prodOrderMain.getStartDate());
        prodProcessExec.setTargetEndDate(prodOrderMain.getEndDate());
        prodProcessExec.setSceneId(sceneId);
        prodProcessExec.setSetupTime(detail.getSetupTime());
        prodProcessExec.setChangeLineTime(detail.getChangeLineTime());
        prodProcessExec.setConstTime(detail.getConstTime());
        prodProcessExec.setPreprocessingTime(detail.getPreprocessingTime());
        prodProcessExec.setPostprocessingTime(detail.getPostprocessingTime());
        prodProcessExec.setId(String.valueOf(SnowFlackIdWorker.getId()));


        prodProcessExec.setConnectProperty(detail.getConnectProperty());


        prodProcessExec.setRoutingId(prodOrderMain.getRoutingId());
        prodProcessExec.setRoutingCode(prodOrderMain.getRoutingCode());
        prodProcessExec.setConnectPropertyName(detail.getConnectPropertyName());


        prodProcessExec.setSpeed(
                Optional.ofNullable(detail.getRuntime())
                        .filter(out -> detail.getSingleOut() != null)
                        .filter(out -> detail.getSingleOut().compareTo(BigDecimal.ZERO) != 0)
                        .map(out -> out.divide(detail.getSingleOut(), 6, RoundingMode.HALF_UP))
                        .map(BigDecimal::doubleValue)  // 关键转换
                        .orElse(0.0)
        );

        return prodProcessExec;
    }



    /**
     ◦ 批量生成工序关联关系

     *
     ◦ @param prodOrderMains 工单ID列表

     ◦ @param sceneId          场景ID

     */
    private void generateProcessRelationsBatch(List<ProdLaunchOrder> prodOrderMains, String sceneId) {
        if (CollectionUtils.isEmpty(prodOrderMains)) {
            log.warn("工单列表为空，跳过工序关系生成");
            return;
        }

        // 先批量获取所有需要的ProdProcessExec对象
        Set<Long> routingDetailIds = new HashSet<>();
        for (ProdLaunchOrder order : prodOrderMains) {
            List<RoutingDetailConnect> connections = routingDetailConnectService.lambdaQuery()
                    .eq(RoutingDetailConnect::getRoutingHeaderId, order.getRoutingId())
                    .eq(RoutingDetailConnect::getIsdeleted, 0)
                    .list();

            for (RoutingDetailConnect connection : connections) {
                routingDetailIds.add(connection.getSourceoperationid());
                routingDetailIds.add(connection.getDestoperationid());
            }
        }

        List<ProdProcessExec> processExecList = prodProcessExecService.lambdaQuery()
                .eq(ProdProcessExec::getSceneId, sceneId)
                .in(ProdProcessExec::getRoutingDetailId, new ArrayList<>(routingDetailIds))
                .list();

        // 构建routingDetailId到execId的映射
        Map<Long, String> routingDetailIdToExecIdMap = processExecList.stream()
                .collect(Collectors.toMap(
                        ProdProcessExec::getRoutingDetailId,
                        ProdProcessExec::getExecId,
                        (existing, replacement) -> existing)); // 处理重复key的情况

        generateProcessRelations(prodOrderMains, sceneId, routingDetailIdToExecIdMap);

        log.info("完成{}个工单的工序关系生成", prodOrderMains.size());
    }

    /**
     ◦ 生成工序关联关系

     *
     ◦ @param prodOrderMains 工单列表

     ◦ @param sceneId        场景ID

     */
    public void generateProcessRelations(List<ProdLaunchOrder> prodOrderMains, String sceneId, Map<Long, String> routingDetailIdToExecIdMap) {
        if (CollectionUtils.isEmpty(prodOrderMains)) {
            return;
        }

        List<ProdOrderProcess> processRelations = new java.util.ArrayList<>();

        for (ProdLaunchOrder prodOrderMain : prodOrderMains) {
            try {
                List<ProdOrderProcess> relations = createProcessRelations(prodOrderMain, sceneId, routingDetailIdToExecIdMap);
                processRelations.addAll(relations);
            } catch (Exception e) {
                log.error("生成工单[{}]工序关系失败", prodOrderMain.getOrderId(), e);
                continue;
            }
        }

        // 批量保存（增加严格校验）
        if (!CollectionUtils.isEmpty(processRelations)) {
            // 过滤掉null元素，并确保关键属性不为空
            List<ProdOrderProcess> validRelations = processRelations.stream()
                    .filter(Objects::nonNull) // 过滤null元素
                    .filter(relation -> {
                        // 根据实际业务规则检查必要属性
                        return relation.getSceneId() != null
                                && relation.getOrderId() != null;

                    })
                    .collect(Collectors.toList());

            if (!CollectionUtils.isEmpty(validRelations)) {
                prodOrderProcessService.saveBatch(validRelations,500);
            } else {
                log.warn("所有工序关系数据无效或为空，跳过批量保存");
            }
        }
    }

    /**
     ◦ 创建工单工序关系

     *
     ◦ @param prodOrderMain 工单信息

     ◦ @param sceneId       场景ID

     ◦ @return 工序关系列表

     */
    private List<ProdOrderProcess> createProcessRelations(ProdLaunchOrder prodOrderMain, String sceneId, Map<Long, String> routingDetailIdToExecIdMap) {
        LambdaQueryWrapper<RoutingDetailConnect> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RoutingDetailConnect::getRoutingHeaderId, prodOrderMain.getRoutingId())
                .eq(RoutingDetailConnect::getIsdeleted, 0); // 添加 isdeleted=0 过滤条件

        List<RoutingDetailConnect> connections = routingDetailConnectService.list(wrapper);

        return connections.stream()
                .map(connection -> createProcessRelation(prodOrderMain, connection, sceneId, routingDetailIdToExecIdMap))
                .collect(Collectors.toList());
    }

    /**
     ◦ 创建单个工序关系

     *
     ◦ @param prodOrderMain 工单信息

     ◦ @param connection    工序连接

     ◦ @param sceneId       场景ID

     ◦ @return 工序关系

     */
    private ProdOrderProcess createProcessRelation(ProdLaunchOrder prodOrderMain,
                                                   RoutingDetailConnect connection,
                                                   String sceneId,
                                                   Map<Long, String> routingDetailIdToExecIdMap) {
        ProdOrderProcess prodOrderProcess = new ProdOrderProcess();
        prodOrderProcess.setSceneId(sceneId);
        prodOrderProcess.setOrderId(prodOrderMain.getOrderId());
        prodOrderProcess.setTaskSeq(connection.getSourceoperation());
        prodOrderProcess.setTargetOrderId(prodOrderMain.getOrderId());
        prodOrderProcess.setTargetTaskSeq(connection.getDestoperation());
        prodOrderProcess.setId(String.valueOf(SnowFlackIdWorker.getId()));

        String execId = routingDetailIdToExecIdMap.get(connection.getSourceoperationid());
        String targetExecId = routingDetailIdToExecIdMap.get(connection.getDestoperationid());

        if (execId != null) {
            prodOrderProcess.setExecId(execId);
        }else {

            prodOrderProcess.setExecId("ttt");
        }

        if (targetExecId != null) {
            prodOrderProcess.setTargetExecId(targetExecId);
        }else {

            prodOrderProcess.setTargetExecId("ttt");
        }

        return prodOrderProcess;
    }

    /**
     ◦ 如果值为null或空白，返回默认值

     *
     ◦ @param value        原始值

     ◦ @param defaultValue 默认值

     ◦ @param <T>          类型

     ◦ @return 处理后的值

     */
    private <T> T defaultIfBlank(T value, T defaultValue) {
        if (value == null) {
            return defaultValue;
        }

        if (value instanceof String && ((String) value).trim().isEmpty()) {
            return defaultValue;
        }

        return value;
    }

    /**
     ◦ 设备对照表

     */
    private void batchInsertEquipMent(
            List<RoutingDetailEquip> routingDetailEquips,
            String sceneId,List<ProdProcessExec> processExecList) {


        if (CollectionUtils.isEmpty(routingDetailEquips)) {
            log.warn("工艺设备列表为空，跳过设备对照表生成");
            return;
        }



        // 创建ProdEquipment列表
        List<ProdEquipment> prodEquipments = new java.util.ArrayList<>();

        // 遍历RoutingDetailEquip列表，转换为ProdEquipment对象
        for (RoutingDetailEquip routingDetailEquip : routingDetailEquips) {
            ProdEquipment prodEquipment = new ProdEquipment();
            prodEquipment.setSceneId(sceneId);
            prodEquipment.setEquipId(routingDetailEquip.getEquipId().longValue());
            // prodEquipment.setEquipCode(routingDetailEquip.getName());
            prodEquipment.setEquipName(routingDetailEquip.getName());
            prodEquipment.setResourceId(routingDetailEquip.getType1());
            prodEquipment.setEfficiencyValue(routingDetailEquip.getEfficiencyValue());
            prodEquipment.setSetupTime(routingDetailEquip.getSetupTime());
            prodEquipment.setId(String.valueOf(SnowFlackIdWorker.getId()));

            prodEquipment.setSpeed(
                    Optional.ofNullable(routingDetailEquip.getDuration())
                            .filter(out -> routingDetailEquip.getOutputQuantity() != null)
                            .filter(out -> routingDetailEquip.getOutputQuantity().compareTo(BigDecimal.ZERO) != 0)
                            .map(out -> out.divide(routingDetailEquip.getOutputQuantity(), 6, RoundingMode.HALF_UP))
                            .map(BigDecimal::doubleValue)  // 关键转换
                            .orElse(0.0)
            );


            String execId = processExecList.stream()
                    .filter(exec -> {
                        Long execRoutingDetailId = exec.getRoutingDetailId();
                        Long equipRoutingDetailId = routingDetailEquip.getRoutingDetailId();

                        // 处理null值
                        if (execRoutingDetailId == null || equipRoutingDetailId == null) {
                            return false;
                        }

                        // 使用equals比较
                        return execRoutingDetailId.equals(equipRoutingDetailId);
                    })
                    .findFirst()
                    .map(ProdProcessExec::getExecId)
                    .orElse(null);
            if (execId != null) {
                prodEquipment.setExecId(execId);
            } else {
                log.warn("未找到routingDetailId={}对应的execId", routingDetailEquip.getRoutingDetailId());
            }

            prodEquipments.add(prodEquipment);
        }

        // 批量保存
        if (!prodEquipments.isEmpty()) {
            prodEquipmentService.saveBatch(prodEquipments);
        }



    }

    /**
     ◦ 获取工艺工序列表

     *
     ◦ @param routingHeaderId 工艺头ID

     ◦ @return 工序列表

     */
    private List<RoutingDetailEquip> getRoutingDetailEquip(Integer routingHeaderId) {


        List<RoutingDetailEquip> detailEquips = routingDetailEquipService.lambdaQuery()
                .eq(RoutingDetailEquip::getRoutingHeaderId, routingHeaderId)
                .eq(RoutingDetailEquip::getIsdeleted, 0)
                .list();// 添加 is_deleted=0 过滤条件


        if (CollectionUtils.isEmpty(detailEquips)) {
            log.error("工艺下无设备信息: {}", routingHeaderId);
            throw new RuntimeException("工艺下无设备信息: " + routingHeaderId);
        }

        return detailEquips;
    }



    /**
     * Copy ProdLaunchOrder data
     */
    private void copyProdLaunchOrders(String oldSceneId, String newSceneId) {
        List<ProdLaunchOrder> oldOrders = prodLaunchOrderService.lambdaQuery()
                .eq(ProdLaunchOrder::getSceneId, oldSceneId)
                .list();

        List<ProdLaunchOrder> newOrders = oldOrders.stream()
                .map(order -> {
                    ProdLaunchOrder newOrder = new ProdLaunchOrder();
                    // Copy all properties except sceneId
                    copyProperties(order, newOrder);
                    newOrder.setSceneId(newSceneId);
                    newOrder.setOrderId(order.getOrderId()); // Generate new ID if needed
                    return newOrder;
                })
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(newOrders)) {
            prodLaunchOrderService.saveBatch(newOrders);
        }
    }

    /**
     * Copy ProdProcessExec data
     */
    private void copyProdProcessExecs(String oldSceneId, String newSceneId) {
        List<ProdProcessExec> oldExecs = prodProcessExecService.lambdaQuery()
                .eq(ProdProcessExec::getSceneId, oldSceneId)
                .list();

        List<ProdProcessExec> newExecs = oldExecs.stream()
                .map(exec -> {
                    ProdProcessExec newExec = new ProdProcessExec();
                    copyProperties(exec, newExec);
                    newExec.setSceneId(newSceneId);
                    newExec.setId(String.valueOf(SnowFlackIdWorker.getId()));
                    return newExec;
                })
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(newExecs)) {
            prodProcessExecService.saveBatch(newExecs);
        }
    }

    /**
     * 复制 ProdOrderProcess
     */
    private void copyProdOrderProcesses(String oldSceneId, String newSceneId) {
        List<ProdOrderProcess> oldProcesses = prodOrderProcessService.lambdaQuery()
                .eq(ProdOrderProcess::getSceneId, oldSceneId)
                .list();

        List<ProdOrderProcess> newProcesses = oldProcesses.stream()
                .map(process -> {
                    ProdOrderProcess newProcess = new ProdOrderProcess();
                    copyProperties(process, newProcess);
                    newProcess.setSceneId(newSceneId);
                    newProcess.setId(String.valueOf(SnowFlackIdWorker.getId()));
                    return newProcess;
                })
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(newProcesses)) {
            prodOrderProcessService.saveBatch(newProcesses);
        }
    }

    /**
     * 复制 ProdEquipment
     */
    private void copyProdEquipments(String oldSceneId, String newSceneId) {
        List<ProdEquipment> oldEquipments = prodEquipmentService.lambdaQuery()
                .eq(ProdEquipment::getSceneId, oldSceneId)
                .list();

        List<ProdEquipment> newEquipments = oldEquipments.stream()
                .map(equipment -> {
                    ProdEquipment newEquipment = new ProdEquipment();
                    copyProperties(equipment, newEquipment);
                    newEquipment.setSceneId(newSceneId);
                    newEquipment.setId(String.valueOf(SnowFlackIdWorker.getId()));
                    return newEquipment;
                })
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(newEquipments)) {
            prodEquipmentService.saveBatch(newEquipments);
        }
    }

    /**
     * 复制 ProdEquipSpecialCal
     */
    private void copyProdEquipSpecialCals(String oldSceneId, String newSceneId) {
        List<ProdEquipSpecialCal> oldCals = prodEquipSpecialCalService.lambdaQuery()
                .eq(ProdEquipSpecialCal::getSceneId, oldSceneId)
                .list();

        List<ProdEquipSpecialCal> newCals = oldCals.stream()
                .map(cal -> {
                    ProdEquipSpecialCal newCal = new ProdEquipSpecialCal();
                    copyProperties(cal, newCal);
                    newCal.setSceneId(newSceneId);
                    newCal.setId(String.valueOf(SnowFlackIdWorker.getId()));
                    return newCal;
                })
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(newCals)) {
            prodEquipSpecialCalService.saveBatch(newCals);
        }
    }

    /**
     * 复制染色体文件（排产结果）
     */
    private void copyChromosomeFile(String oldSceneId, String newSceneId) {
        try {
            // 检查源场景是否有染色体文件


            if (sceneService.chromosomeFileExists(oldSceneId)) {
                // 从源场景加载染色体
                Chromosome chromosome = sceneService.loadChromosomeFromFile(oldSceneId);
                if (chromosome != null) {
                    // 保存到新场景
                    boolean saved = sceneService.saveChromosomeToFile(chromosome, newSceneId);
                    if (saved) {
                        log.info("成功复制染色体文件，从场景 {} 到场景 {}", oldSceneId, newSceneId);
                    } else {
                        log.warn("复制染色体文件失败，从场景 {} 到场景 {}", oldSceneId, newSceneId);
                    }
                }
            } else {
                log.info("源场景 {} 没有染色体文件，无需复制", oldSceneId);
            }
        } catch (Exception e) {
            log.error("复制染色体文件时发生异常，从场景 {} 到场景 {}", oldSceneId, newSceneId, e);
        }
    }


}