package com.aps.config;

import io.swagger.v3.oas.models.media.MapSchema;
import io.swagger.v3.oas.models.media.Schema;
import io.swagger.v3.oas.models.media.StringSchema;
import io.swagger.v3.oas.models.examples.Example; // 添加这个导入
import org.springdoc.core.customizers.OperationCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.method.HandlerMethod;

import java.util.HashMap;
import java.util.Map;

@Configuration
public class SwaggerMapParamConfig {

    @Bean
    public OperationCustomizer customizeOperation() {
        return (operation, handlerMethod) -> {
            if (operation.getRequestBody() != null &&
                    operation.getRequestBody().getContent() != null &&
                    operation.getRequestBody().getContent().get("application/json") != null) {

                Schema<?> schema = operation.getRequestBody().getContent().get("application/json").getSchema();
                if (schema instanceof MapSchema) {
                    addParamExamples(operation, handlerMethod);
                }
            }
            return operation;
        };
    }

    private void addParamExamples(io.swagger.v3.oas.models.Operation operation, HandlerMethod handlerMethod) {
        String methodName = handlerMethod.getMethod().getName();
        Map<String, Schema> properties = new HashMap<>();
        Map<String, Example> examples = new HashMap<>(); // 修改这里：Object -> Example

        switch (methodName) {
            case "lanuch":
                properties.put("sceneName", new StringSchema().description("场景名称").example("scene_001"));
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("启动工单示例", createExample(
                        "启动指定场景的工单处理",
                        "{\n" +
                                "    \"sceneName\": \"scene_001\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;

            case "schedule":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("排程计算示例", createExample(
                        "执行场景的排程算法计算",
                        "{\n" +
                                "    \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\"\n" +
                                "}"
                ));
                break;

            case "copyScene":
                properties.put("oldSceneId", new StringSchema().description("原场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("newSceneId", new StringSchema().description("新场景名称").example("scene_002"));
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("完整复制示例", createExample(
                        "复制场景包含所有数据和配置",
                        "{\n" +
                                "    \"oldSceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "    \"newSceneName\": \"scene_002\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                examples.put("最小复制示例", createExample(
                        "仅提供必需参数的场景复制",
                        "{\n" +
                                "    \"oldSceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "    \"newSceneName\": \"scene_002\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;

            case "exportPlan":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("导出计划示例", createExample(
                        "导出指定场景的排程结果为生产计划",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                break;

            case "deleteScene":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("删除场景示例", createExample(
                        "删除指定的场景",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                examples.put("批量删除示例", createExample(
                        "删除多个场景（如果支持批量删除）",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                break;

            case "getAllScene":
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("获取场景示例", createExample(
                        "获取指定用户的所有场景",
                        "{\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                examples.put("管理员查询示例", createExample(
                        "管理员查询系统所有场景（如果支持）",
                        "{\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;


            case "getResourceGantt":
            case "getProductGantt":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                examples.put("示例", createExample(
                        "请求示例",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\"\n" +
                                "}"
                ));
                break;
                
            case "operationMove":
                properties.put("SceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("opid", new StringSchema().description("操作ID").example(1));
                properties.put("newStartTime", new StringSchema().description("新的开始时间").example("2025-11-03T07:36:00.000Z"));
                properties.put("newMachineId", new StringSchema().description("新机器ID").example(3402));
                examples.put("移动操作示例", createExample(
                        "将指定操作移动到新的时间和机器上",
                        "{\n" +
                                "  \"SceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"opid\": 1,\n" +
                                "  \"newStartTime\": \"2025-11-03T07:36:00.000Z\",\n" +
                                "  \"newMachineId\": 3402\n" +
                                "}"
                ));
                break;
                
            case "operationedit":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("operation", new StringSchema().description("操作对象"));
                examples.put("编辑操作示例", createExample(
                        "编辑指定的操作",
                        "{\n" +
                                "  \"SceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"operation\": {}\n" +
                                "}"
                ));
                break;
                
            case "changebasetime":
                properties.put("SceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("BaseTime", new StringSchema().description("基础时间").example("2025-11-03T07:36:00.000Z"));
                examples.put("更改基准时间示例", createExample(
                        "更改场景的基准时间",
                        "{\n" +
                                "  \"SceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"BaseTime\": \"2025-11-03T07:36:00.000Z\"\n" +
                                "}"
                ));
                break;
                
            case "spiltOperation":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("opid", new StringSchema().description("操作ID").example(1));
                properties.put("splitCounts", new StringSchema().description("拆分数量数组").example("[20000.0, 30000.0]"));
                examples.put("拆分操作示例", createExample(
                        "按指定数量拆分操作",
                        "{\n" +
                                "  \"SceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"opid\": 2,\n" +
                                "  \"splitCounts\": [20000, 30000]\n" +
                                "}"
                ));
                break;
                
            case "delOperation":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("opid", new StringSchema().description("操作ID").example(1));
                examples.put("删除操作示例", createExample(
                        "删除指定操作",
                        "{\n" +
                                "  \"SceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"opid\": 1\n" +
                                "}"
                ));
                break;
                
            case "lockedOperation":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("opid", new StringSchema().description("操作ID").example(1));
                properties.put("isLocked", new StringSchema().description("是否锁定").example(true));
                examples.put("锁定操作示例", createExample(
                        "锁定或解锁指定操作",
                        "{\n" +
                                "  \"SceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"opid\": 1,\n" +
                                "  \"isLocked\": true\n" +
                                "}"
                ));
                break;
                
            case "spiltOrder":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("orderid", new StringSchema().description("订单ID").example("fcc0892a-0483-4da7-8414-9ce98be36e53"));
                properties.put("splitCounts", new StringSchema().description("拆分数量数组").example("[20000.0, 20000.0, 10000.0]"));
                examples.put("拆分订单示例", createExample(
                        "按指定数量拆分订单",
                        "{\n" +
                                "  \"SceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"orderid\": \"fcc0892a-0483-4da7-8414-9ce98be36e53\",\n" +
                                "  \"splitCounts\": [20000, 20000, 10000]\n" +
                                "}"
                ));
                break;
                
            case "orderMerge":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("sourceorderid", new StringSchema().description("源订单ID").example("5be078b725b34ade8b6638f74dad6b10"));
                properties.put("targetorderid", new StringSchema().description("目标订单ID").example("2a0f23d2429f4e5da7b3929da75a803d"));
                examples.put("合并订单示例", createExample(
                        "将源订单合并到目标订单",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"sourceorderid\": \"5be078b725b34ade8b6638f74dad6b10\",\n" +
                                "  \"targetorderid\": \"2a0f23d2429f4e5da7b3929da75a803d\"\n" +
                                "}"
                ));
                break;
        }

        if (!properties.isEmpty()) {
            MapSchema mapSchema = new MapSchema();
            mapSchema.setProperties(properties);
            mapSchema.setDescription(getMethodDescription(methodName));

            if (!examples.isEmpty()) {
                mapSchema.setExample(examples.values().iterator().next());
                operation.getRequestBody().getContent().get("application/json").setExamples(examples); // 这行现在类型匹配了
            }

            operation.getRequestBody().getContent().get("application/json").setSchema(mapSchema);
        }
    }

    // 修改这里：返回类型从 Map<String, Object> 改为 Example
    private Example createExample(String summary, String value) {
        Example example = new Example();
        example.setSummary(summary);
        example.setValue(value);
        return example;
    }

    private String getMethodDescription(String methodName) {
        switch (methodName) {
            case "lanuch":
                return "启动工单请求参数";
            case "schedule":
                return "排程计算请求参数";
            case "copyScene":
                return "复制场景请求参数";
            case "exportPlan":
                return "导出计划请求参数";
            case "deleteScene":
                return "删除场景请求参数";
            case "getAllScene":
                return "获取场景列表请求参数";

            case "getResourceGantt":

            case "getProductGantt":
                return "获取资源甘特图请求参数";
            case "operationMove":
                return "移动操作请求参数";
            case "operationedit":
                return "编辑操作请求参数";
            case "changebasetime":
                return "更改基准时间请求参数";
            case "spiltOperation":
                return "拆分操作请求参数";
            case "delOperation":
                return "删除操作请求参数";
            case "lockedOperation":
                return "锁定操作请求参数";
            case "spiltOrder":
                return "拆分订单请求参数";
            case "orderMerge":
                return "合并订单请求参数";
            default:
                return "请求参数";
        }
    }
}