package com.aps.service.Algorithm;

import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Algorithm.ObjectiveWeights;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * 增强版NSGA-II工具类（自定义权重+并行非支配排序）
 */
public class NSGAIIUtils {
    private ObjectiveWeights objectiveWeights;
    private int taskCount; // 并行任务数（默认=CPU核心数）

    /**
     * 初始化（设置权重和并行任务数）
     */
    public void init(ObjectiveWeights weights, int taskCount) {
        this.objectiveWeights = (weights != null) ? weights : new ObjectiveWeights();
        this.taskCount = (taskCount > 0) ? taskCount : Runtime.getRuntime().availableProcessors();
    }

    /**
     * 重载初始化方法（默认任务数）
     */
    public void init(ObjectiveWeights weights) {
        init(weights, 0);
    }


    /**
     * 并行快速非支配排序（大规模种群提速50%+）
     */
    public List<List<Chromosome>> parallelFastNonDominatedSort(List<Chromosome> population) {
        int popSize = population.size();


        if (popSize <= taskCount * 10) { // 小规模种群直接串行
            // 步骤1：预处理 - 计算归一化目标值和加权目标值
            preprocessObjectives(population);
            return fastNonDominatedSort(population);
        }

        // 步骤1：预处理 - 计算归一化目标值和加权目标值
        preprocessObjectives(population);

        // 步骤2：拆分种群为多个块，并行计算支配关系
        List<List<Chromosome>> chunks = splitPopulation(population, taskCount);
        ConcurrentHashMap<Chromosome, ConcurrentLinkedQueue<Chromosome>> dominanceMatrix = new ConcurrentHashMap<>();
        ConcurrentHashMap<Chromosome, AtomicInteger> dominatedCount = new ConcurrentHashMap<>();

        // 初始化字典
        for (Chromosome chromo : population) {
            dominanceMatrix.putIfAbsent(chromo, new ConcurrentLinkedQueue<>());
            dominatedCount.putIfAbsent(chromo, new AtomicInteger(0));
        }

        // 并行计算支配关系
        chunks.parallelStream().forEach(chunk -> {
            for (Chromosome p : chunk) {
                for (Chromosome q : population) {
                    if (p == q) continue;

                    if (dominates(p, q)) {
                        dominanceMatrix.get(p).add(q);
                    } else if (dominates(q, p)) {
                        dominatedCount.get(p).incrementAndGet();
                    }
                }
            }
        });

        // 步骤3：生成前沿
        List<List<Chromosome>> fronts = new ArrayList<>();
        List<Chromosome> front1 = new ArrayList<>();

        for (Chromosome p : population) {
            if (dominatedCount.get(p).get() == 0) {
                p.setRank(1);
                front1.add(p);
            }
        }

        fronts.add(front1);
        int i = 0;

        while (fronts.get(i).size() > 0) {
            List<Chromosome> nextFront = new ArrayList<>();
            for (Chromosome p : fronts.get(i)) {
                for (Chromosome q : dominanceMatrix.get(p)) {
                    AtomicInteger count = dominatedCount.get(q);
                    if (count.decrementAndGet() == 0) {
                        q.setRank(i + 2);
                        nextFront.add(q);
                    }
                }
            }
            i++;
            if (!nextFront.isEmpty()) {
                fronts.add(nextFront);
            }
        }

        return fronts;
    }



    /**
     * 串行快速非支配排序（兼容小规模种群）
     */
    public List<List<Chromosome>> fastNonDominatedSort(List<Chromosome> population) {
        // 1. 初始化数据结构：S（被支配集合）、n（支配数）
        Map<Chromosome, List<Chromosome>> S = new HashMap<>();
        Map<Chromosome, Integer> n = new HashMap<>();
        List<List<Chromosome>> fronts = new ArrayList<>();

        // 为每个个体预先初始化 S 和 n
        for (Chromosome p : population) {
            S.put(p, new ArrayList<>()); // 初始化空集合，避免null
            n.put(p, 0); // 支配数初始化为0
        }

        // 2. 计算每个个体的支配关系
        for (Chromosome p : population) {
            for (Chromosome q : population) {
                if (p == q) continue;
               // p.getWeightedObjective() < q.getWeightedObjective();
                if (dominates(p, q)) {
                    // p支配q，将q加入p的被支配集合S
                    S.get(p).add(q);
                } else if (dominates(q, p)) {
                    // q支配p，增加p的支配数n
                    n.put(p, n.get(p) + 1);
                }
            }

            // 3. 初始化第一前沿（支配数为0的个体）
            if (n.get(p) == 0) {
                if (fronts.isEmpty()) {
                    fronts.add(new ArrayList<>());
                }
                fronts.get(0).add(p);
            }
        }

        // 4. 生成后续前沿
        int i = 0;
        while (!fronts.get(i).isEmpty()) {
            List<Chromosome> nextFront = new ArrayList<>();
            for (Chromosome p : fronts.get(i)) {
                // 遍历p的被支配集合S.get(p)（此时已初始化，不会为null）
                for (Chromosome q : S.get(p)) {
                    n.put(q, n.get(q) - 1);
                    if (n.get(q) == 0) {
                        nextFront.add(q);
                    }
                }
            }
            i++;
            fronts.add(nextFront);
        }

        return fronts;
    }
    /**
     * 预处理目标值（归一化+加权）
     */
    private void preprocessObjectives(List<Chromosome> population) {


        // 计算每个目标的最大/最小值
        double[] minValues = new double[5];
        double[] maxValues = new double[5];
        for (int i = 0; i < 5; i++) {
            final int idx = i;
            minValues[i] = population.stream().mapToDouble(c -> c.getObjectives()[idx]).min().orElse(0);
            maxValues[i] = population.stream().mapToDouble(c -> c.getObjectives()[idx]).max().orElse(0);
        }
        boolean[] isMinimize = {true, true, true, false, true};
        // 并行计算归一化和加权目标值
        population.parallelStream().forEach(chromo -> {
            double[] normalized = objectiveWeights.normalizeObjectives(chromo.getObjectives(), minValues, maxValues,isMinimize);
            if (objectiveWeights.isPureNSGAIIMode()) {
                chromo.setWeightedObjective(objectiveWeights.calculateObjective(normalized));
            }else {
                chromo.setWeightedObjective(objectiveWeights.calculateWeightedObjective(normalized));
            }
        });
    }

    /**
     * 拆分种群为多个块（用于并行计算）
     */
    private List<List<Chromosome>> splitPopulation(List<Chromosome> population, int chunkCount) {
        List<List<Chromosome>> chunks = new ArrayList<>();
        int chunkSize = (int) Math.ceil((double) population.size() / chunkCount);

        for (int i = 0; i < population.size(); i += chunkSize) {
            int end = Math.min(i + chunkSize, population.size());
            chunks.add(population.subList(i, end));
        }

        return chunks;
    }



    /**
     * 判断p是否支配q（支持自定义权重）
     */
    public boolean dominates(Chromosome p, Chromosome q) {
        // 纯NSGA-II模式：标准帕累托支配
        if (objectiveWeights.isPureNSGAIIMode()) {
            boolean allBetterOrEqual = true;
            boolean oneBetter = false;

            double[] pObjectives = p.getObjectives();
            double[] qObjectives = q.getObjectives();

            for (int i = 0; i < pObjectives.length; i++) {
                if (pObjectives[i] > qObjectives[i]) {
                    allBetterOrEqual = false;
                    break;
                } else if (pObjectives[i] < qObjectives[i]) {
                    oneBetter = true;
                }
            }

            return allBetterOrEqual && oneBetter;
        }
        // 加权NSGA-II模式：融合权重的支配关系
        else {
            // 方式1：加权目标值更小则支配（简单高效）
            return p.getWeightedObjective() < q.getWeightedObjective();

            // 方式2（可选）：加权后的帕累托支配
            // double pWeightedSum = 0, qWeightedSum = 0;
            // for (int i = 0; i < p.getObjectives().length; i++) {
            //     pWeightedSum += p.getObjectives()[i] * objectiveWeights.getWeights()[i];
            //     qWeightedSum += q.getObjectives()[i] * objectiveWeights.getWeights()[i];
            // }
            // return pWeightedSum < qWeightedSum;
        }
    }

    /**
     * 并行计算拥挤距离
     */
    public void parallelCalculateCrowdingDistance(List<Chromosome> front) {
        if (front.isEmpty()) return;

        int objCount = front.get(0).getObjectives().length;
        int popSize = front.size();

        // 初始化距离为0
        front.parallelStream().forEach(chromo -> chromo.setCrowdingDistance(0.0));

        // 对每个目标维度并行计算距离
        IntStream.range(0, objCount).parallel().forEach(m -> {
            // 按当前目标排序
            List<Chromosome> sorted = front.stream()
                    .sorted(Comparator.comparingDouble(c -> c.getObjectives()[m]))
                    .collect(Collectors.toList());

            // 边界个体距离设为无穷大
            sorted.get(0).setCrowdingDistance(Double.POSITIVE_INFINITY);
            sorted.get(popSize - 1).setCrowdingDistance(Double.POSITIVE_INFINITY);

            double maxObj = sorted.get(popSize - 1).getObjectives()[m];
            double minObj = sorted.get(0).getObjectives()[m];

            if (maxObj - minObj == 0) return;

            // 计算中间个体的拥挤距离
            for (int i = 1; i < popSize - 1; i++) {
                Chromosome chromo = sorted.get(i);
                synchronized (chromo) { // 线程安全
                    double distance = chromo.getCrowdingDistance() +
                            (sorted.get(i + 1).getObjectives()[m] - sorted.get(i - 1).getObjectives()[m])
                                    / (maxObj - minObj);
                    chromo.setCrowdingDistance(distance);
                }
            }
        });
    }

    /**
     * 选择下一代种群（兼容并行计算）
     */
    public List<Chromosome> selectNextPopulation(List<List<Chromosome>> fronts, int populationSize) {
        List<Chromosome> nextPopulation = new ArrayList<>();
        int currentSize = 0;

        for (List<Chromosome> front : fronts) {
            if (currentSize + front.size() <= populationSize) {
                nextPopulation.addAll(front);
                currentSize += front.size();
            } else {
                // 并行计算拥挤距离
                parallelCalculateCrowdingDistance(front);
                // 按拥挤距离排序，取前N个
                List<Chromosome> sortedByDistance = front.stream()
                        .sorted((c1, c2) -> Double.compare(c2.getCrowdingDistance(), c1.getCrowdingDistance()))
                        .collect(Collectors.toList());
                nextPopulation.addAll(sortedByDistance.subList(0, populationSize - currentSize));
                currentSize = populationSize;
                break;
            }
        }

        return nextPopulation;
    }




}
