package com.aps.controller.codeGeneratorController;


import com.aps.entity.codeGeneratorEntity.DatabaseConfig;
import com.aps.entity.codeGeneratorEntity.GenerateRequest;
import com.aps.service.codeGeneratorService.CodeGeneratorService;
import com.aps.service.codeGeneratorService.DatabaseConfigService;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/api/code-generator")
@Tag(name = "代码生成器", description = "代码生成相关接口")
public class CodeGeneratorController {

    @Autowired
    private CodeGeneratorService codeGeneratorService;

    @Autowired
    private DatabaseConfigService databaseConfigService;

    @GetMapping("/config")
    @Operation(summary = "获取当前主数据源配置", description = "获取当前主数据源配置")
    public DatabaseConfig getDatabaseConfig() {
        DatabaseConfig config = databaseConfigService.getPrimaryDatabaseConfig();
        if (config == null) {
            throw new RuntimeException("无法获取数据库配置，请检查application.yml配置");
        }
        return config;
    }

    @GetMapping("/config/{dataSourceName}")
    @Operation(summary = "获取指定数据源配置", description = "获取指定数据源配置")
    public DatabaseConfig getDatabaseConfig(@PathVariable String dataSourceName) {
        DatabaseConfig config = databaseConfigService.getDatabaseConfig(dataSourceName);
        if (config == null) {
            throw new RuntimeException("数据源 " + dataSourceName + " 配置不存在或配置错误");
        }
        return config;
    }

    @GetMapping("/tables/{dataSourceName}")
    @Operation(summary = "获取指定数据源的表列表", description = "获取指定数据源的表列表")
    public List<String> getTables(@PathVariable String dataSourceName) {
        DatabaseConfig config = databaseConfigService.getDatabaseConfig(dataSourceName);
        if (config == null || config.getUrl() == null) {
            throw new RuntimeException("数据源 " + dataSourceName + " 配置错误，无法获取表列表");
        }
        return codeGeneratorService.getTableList(config);
    }

    @GetMapping("/test-connection/{dataSourceName}")
    @Operation(summary = "测试指定数据源连接", description = "测试指定数据源连接")
    public String testConnection(@PathVariable String dataSourceName) {
        DatabaseConfig config = databaseConfigService.getDatabaseConfig(dataSourceName);
        if (config == null) {
            return dataSourceName + " 数据源配置不存在";
        }
        
        boolean isConnected = databaseConfigService.testConnection(config);
        return isConnected ? dataSourceName + " 数据库连接成功！" : dataSourceName + " 数据库连接失败！";
    }

    @PostMapping("/generate/{dataSourceName}")
    @Operation(summary = "使用指定数据源生成代码", description = "使用指定数据源生成代码")
    public String generateCode(@PathVariable String dataSourceName, 
                               @RequestBody GenerateRequest request) {
        try {
            DatabaseConfig dbConfig = databaseConfigService.getDatabaseConfig(dataSourceName);
            if (dbConfig == null || dbConfig.getUrl() == null) {
                return "数据源 " + dataSourceName + " 配置错误，请检查application.yml";
            }
            
            codeGeneratorService.generateCode(dbConfig, request.getTables(), 
                request.getPackageName(), request.getAuthor());
            return "代码生成成功！(数据源: " + dataSourceName + ")";
        } catch (Exception e) {
            return "代码生成失败: " + e.getMessage();
        }
    }
}