package com.aps.service.Algorithm;

import com.aps.common.util.ProductionDeepCopyUtil;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Algorithm.GAScheduleResult;
import com.aps.entity.Algorithm.GlobalOperationInfo;
import com.aps.entity.basic.Entry;
import com.aps.entity.basic.GlobalParam;
import com.aps.entity.basic.MachineOption;
import com.aps.service.plan.MachineSchedulerService;

import javax.xml.transform.Result;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;

/**
 * 作者：佟礼
 * 时间：2025-11-30
 */
public class ScheduleOperationService {
    /**
        * 移动工序方法
     * @param chromosome 染色体对象
     * @param opId 工序ID
     * @param newStartTime 新开始时间
     * @param newMachineId 新设备ID
     * @param baseTime 基准时间
     */
    public void moveOperation(Chromosome chromosome, int opId, int newStartTime,
                              int newMachineId, LocalDateTime baseTime, GlobalParam globalParam) {
        List<Entry> allOperations=chromosome.getAllOperations();
        // 获取目标结果和工序
        GAScheduleResult targetResult = chromosome.getResult().stream()
                .filter(r -> r.getOperationId() == opId)
                .findFirst()
                .orElseThrow(() -> new NoSuchElementException("Operation not found: " + opId));

        Entry targetOp = allOperations.stream()
                .filter(o -> o.getId() == opId)
                .findFirst()
                .orElseThrow(() -> new NoSuchElementException("Operation not found: " + opId));

        int machineOptionIndex = targetOp.getMachineOptions().stream()
                .map(MachineOption::getMachineId)
                .collect(Collectors.toList())
                .indexOf(newMachineId) + 1;

        if(machineOptionIndex==0)
        {
            throw new NoSuchElementException("Machine not found: " + newMachineId);
        }

        // 设置约束
        targetResult.setDesignatedStartTime(newStartTime);
        targetResult.setForcedMachineId(newMachineId);

        // 更新设备选择序列
        int globalOpIndex = chromosome.getGlobalOpList().stream()
                .filter(g -> g.getOp().getId() == opId)
                .findFirst()
                .map(GlobalOperationInfo::getGlobalOpId)
                .orElseThrow(() -> new NoSuchElementException("Global operation not found: " + opId));





        chromosome.getMachineSelection().set(globalOpIndex, machineOptionIndex);

        // 生成新的工序顺序
        Map<Integer, Integer> opTimeMap = chromosome.getResult().stream()
                .collect(Collectors.toMap(
                        GAScheduleResult::getOperationId,
                        r -> r.getOperationId() == opId ? newStartTime : r.getStartTime()
                ));

        List<Integer> operationSequencing = allOperations.stream()
                .sorted((op1, op2) -> {
                    int time1 = opTimeMap.getOrDefault(op1.getId(), Integer.MAX_VALUE);
                    int time2 = opTimeMap.getOrDefault(op2.getId(), Integer.MAX_VALUE);
                    if (time1 != time2) {
                        return Integer.compare(time1, time2);
                    } else {
                        return Integer.compare(op1.getSequence(), op2.getSequence());
                    }
                })
                .map(Entry::getGroupId)
                .collect(Collectors.toList());

        chromosome.setOperationSequencing(operationSequencing);

        // 优化解码
        MachineSchedulerService machineScheduler = new MachineSchedulerService(baseTime);
        GeneticDecoder decoder = new GeneticDecoder(globalParam,baseTime, chromosome.getInitMachines(),
                chromosome.getOrders(), null, machineScheduler);
        chromosome.setMachines(chromosome.getInitMachines());
        chromosome.setResultOld(ProductionDeepCopyUtil.deepCopyList(chromosome.getResult()));
        chromosome.getResult().clear();

        decoder.decode(chromosome);
        if(chromosome.getFitness()==0) {
            FitnessCalculator fitnessCalc = new FitnessCalculator();
            chromosome.setFitness(fitnessCalc.calculateFitness(chromosome));
        }
    }
}
