package com.aps.common.util;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * BigDecimal 运算工具类
 */
public class BigDecimalUtil {
    
    // 默认精度（小数点后4位）
    private static final int DEFAULT_SCALE = 4;
    // 默认舍入模式
    private static final RoundingMode DEFAULT_ROUNDING_MODE = RoundingMode.HALF_UP;
    
    /**
     * 安全加法（处理null值）
     */
    public static BigDecimal safeAdd(BigDecimal num1, BigDecimal num2) {
        num1 = defaultIfNull(num1);
        num2 = defaultIfNull(num2);
        return num1.add(num2);
    }
    
    /**
     * 安全减法
     */
    public static BigDecimal safeSubtract(BigDecimal num1, BigDecimal num2) {
        num1 = defaultIfNull(num1);
        num2 = defaultIfNull(num2);
        return num1.subtract(num2);
    }
    
    /**
     * 安全乘法
     */
    public static BigDecimal safeMultiply(BigDecimal num1, BigDecimal num2) {
        num1 = defaultIfNull(num1);
        num2 = defaultIfNull(num2);
        return num1.multiply(num2);
    }
    
    /**
     * 安全乘法（带精度）
     */
    public static BigDecimal safeMultiply(BigDecimal num1, BigDecimal num2, int scale) {
        BigDecimal result = safeMultiply(num1, num2);
        return result.setScale(scale, DEFAULT_ROUNDING_MODE);
    }
    
    /**
     * 安全除法
     */
    public static BigDecimal safeDivide(BigDecimal numerator, BigDecimal denominator) {
        return safeDivide(numerator, denominator, DEFAULT_SCALE);
    }
    
    /**
     * 安全除法（指定精度）
     */
    public static BigDecimal safeDivide(BigDecimal numerator, BigDecimal denominator, int scale) {
        numerator = defaultIfNull(numerator);
        denominator = defaultIfNull(denominator);
        
        if (denominator.compareTo(BigDecimal.ZERO) == 0) {
            throw new ArithmeticException("除数不能为零");
        }
        
        return numerator.divide(denominator, scale, DEFAULT_ROUNDING_MODE);
    }
    
    /**
     * 比较大小
     */
    public static boolean isGreaterThan(BigDecimal num1, BigDecimal num2) {
        num1 = defaultIfNull(num1);
        num2 = defaultIfNull(num2);
        return num1.compareTo(num2) > 0;
    }
    
    public static boolean isLessThan(BigDecimal num1, BigDecimal num2) {
        num1 = defaultIfNull(num1);
        num2 = defaultIfNull(num2);
        return num1.compareTo(num2) < 0;
    }
    
    public static boolean equals(BigDecimal num1, BigDecimal num2) {
        num1 = defaultIfNull(num1);
        num2 = defaultIfNull(num2);
        return num1.compareTo(num2) == 0;
    }
    
    /**
     * 处理null值
     */
    private static BigDecimal defaultIfNull(BigDecimal value) {
        return value != null ? value : BigDecimal.ZERO;
    }
    

}