package com.aps.service.common;

import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.common.Paged;
import com.aps.service.plan.SceneService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Chromosome数据服务类
 * 提供对Chromosome对象中各种实体数据的查询服务
 */
@Service
public class ChromosomeDataService {

    @Autowired
    private SceneService sceneService;

    /**
     * 根据场景ID和实体名称查询Chromosome中的数据
     *
     * @param sceneId 场景ID
     * @param entityName 实体名称 (如: order, entry, machine等)
     * @return 对应的数据对象
     */
    public Object queryChromosomeData(String sceneId, String entityName) {
        // 从文件中加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            throw new RuntimeException("未找到场景ID为 " + sceneId + " 的Chromosome数据");
        }

        try {
            // 根据实体名称映射到Chromosome中的字段名
            String fieldName = mapEntityToField(entityName);

            // 使用反射获取字段值
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);

            return field.get(chromosome);
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + entityName, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }

    /**
     * 根据场景ID和实体名称查询Chromosome中的分页数据
     *
     * @param sceneId 场景ID
     * @param entityName 实体名称
     * @param page 页码(从1开始)
     * @param size 每页大小
     * @return 分页数据
     */
    public Map<String, Object> queryChromosomeDataWithPagination(String sceneId, String entityName, int page, int size) {
        Object data = queryChromosomeData(sceneId, entityName);

        Map<String, Object> result = new HashMap<>();

        // 处理分页逻辑（如果数据是List类型）
        if (data instanceof List) {
            List<?> dataList = (List<?>) data;
            int total = dataList.size();
            int fromIndex = (page - 1) * size;
            // 确保起始索引不超出范围
            fromIndex = Math.min(fromIndex, total);
            int toIndex = Math.min(fromIndex + size, total);

            // 分页数据
            List<?> pagedData = dataList.subList(fromIndex, toIndex);

            result.put("records", pagedData);
            result.put("total", total);
            result.put("current", page);
            result.put("size", size);
        } else {
            result.put("records", data);
            result.put("total", 1);
            result.put("current", 1);
            result.put("size", 1);
        }

        return result;
    }

    /**
     * 根据场景ID和实体名称查询Chromosome中的分页数据，带条件过滤
     *
     * @param sceneId 场景ID
     * @param entityName 实体名称
     * @param paged 分页和条件对象
     * @return 分页数据
     */
    public Map<String, Object> queryChromosomeDataWithConditions(String sceneId, String entityName, Paged paged) {
        Object data = queryChromosomeData(sceneId, entityName);

        Map<String, Object> result = new HashMap<>();
        int page = paged.getPageIndex() != null ? paged.getPageIndex() : 1;
        int size = paged.getPageSize() != null ? paged.getPageSize() : 10;

        // 处理分页和条件逻辑（如果数据是List类型）
        if (data instanceof List) {
            List<?> dataList = (List<?>) data;
            
            // 应用条件过滤
            if (!CollectionUtils.isEmpty(paged.getConditions())) {
                dataList = filterDataByConditions(dataList, paged.getConditions());
            }
            
            int total = dataList.size();
            int fromIndex = (page - 1) * size;
            // 确保起始索引不超出范围
            fromIndex = Math.min(fromIndex, total);
            int toIndex = Math.min(fromIndex + size, total);

            // 分页数据
            List<?> pagedData = dataList.subList(fromIndex, toIndex);

            result.put("records", pagedData);
            result.put("total", total);
            result.put("current", page);
            result.put("size", size);
        } else {
            result.put("records", data);
            result.put("total", 1);
            result.put("current", 1);
            result.put("size", 1);
        }

        return result;
    }

    /**
     * 根据条件过滤数据
     * @param dataList 原始数据列表
     * @param conditions 条件列表
     * @return 过滤后的数据列表
     */
    private List<?> filterDataByConditions(List<?> dataList, List<com.aps.entity.common.ConditionEntity> conditions) {
        // 简单实现：按字段相等条件过滤
        List<?> filteredList = dataList;
        
        for (com.aps.entity.common.ConditionEntity condition : conditions) {
            String fieldName = condition.getFieldName();
            String fieldValue = condition.getFieldValue();
            com.aps.entity.common.ConditionEnum conditionType = 
                com.aps.entity.common.ConditionEnum.getByName(condition.getConditionalType());
            
            if (conditionType == com.aps.entity.common.ConditionEnum.Equal) {
                filteredList = filteredList.stream()
                    .filter(item -> {
                        try {
                            Field field = item.getClass().getDeclaredField(fieldName);
                            field.setAccessible(true);
                            Object value = field.get(item);
                            return value != null && value.toString().equals(fieldValue);
                        } catch (Exception e) {
                            // 字段不存在或访问异常，跳过该条件
                            return true;
                        }
                    })
                    .collect(Collectors.toList());
            }
            // 可以继续添加其他条件类型的处理
        }
        
        return filteredList;
    }

    /**
     * 将实体名称映射到Chromosome中的字段名
     * @param entityName 实体名称
     * @return Chromosome中的字段名
     */
    private String mapEntityToField(String entityName) {
        switch (entityName.toLowerCase()) {
            case "order":
                return "orders";
            case "entry":
                return "allOperations";
            case "machine":
                return "InitMachines";
            case "globaloperationinfo":
                return "globalOpList";
            case "groupresult":
                return "OperatRel";
            default:
                return entityName; // 如果名称相同则直接返回
        }
    }
    
    /**
     * 更新Chromosome中的数据
     * @param sceneId 场景ID
     * @param entityName 实体名称
     * @param data 更新的数据
     * @return 是否更新成功
     */
    public boolean updateChromosomeData(String sceneId, String entityName, Map<String, Object> data) {
        // 从文件中加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            throw new RuntimeException("未找到场景ID为 " + sceneId + " 的Chromosome数据");
        }
        
        try {
            // 根据实体名称映射到Chromosome中的字段名
            String fieldName = mapEntityToField(entityName);
            
            // 使用反射设置字段值
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);
            
            // 获取原始数据
            Object originalData = field.get(chromosome);
            
            // 如果原始数据是List类型，并且传入的data包含id字段，则更新列表中的特定项
            if (originalData instanceof List && data.containsKey("id")) {
                List<Object> dataList = (List<Object>) originalData;
                String id = data.get("id").toString();
                
                // 查找并更新匹配的项
                for (int i = 0; i < dataList.size(); i++) {
                    Object item = dataList.get(i);
                    try {
                        Field idField = item.getClass().getDeclaredField("id");
                        idField.setAccessible(true);
                        Object itemId = idField.get(item);
                        if (itemId != null && itemId.toString().equals(id)) {
                            // 找到匹配项，更新它
                            updateObjectFields(item, data);
                            break;
                        }
                    } catch (Exception e) {
                        // 如果没有id字段或访问失败，尝试使用Id字段
                        try {
                            Field idField = item.getClass().getDeclaredField("Id");
                            idField.setAccessible(true);
                            Object itemId = idField.get(item);
                            if (itemId != null && itemId.toString().equals(id)) {
                                // 找到匹配项，更新它
                                updateObjectFields(item, data);
                                break;
                            }
                        } catch (Exception ex) {
                            // 忽略异常，继续下一个
                        }
                    }
                }
            } else {
                // 直接更新整个字段
                field.set(chromosome, data);
            }
            
            // 保存更新后的Chromosome到文件
            return sceneService.saveChromosomeToFile(chromosome, sceneId);
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + entityName, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 批量操作Chromosome中的数据
     * @param sceneId 场景ID
     * @param entityName 实体名称
     * @param dataList 批量数据
     * @return 操作成功的数量
     */
    public int batchChromosomeData(String sceneId, String entityName, List<Map<String, Object>> dataList) {
        // 从文件中加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            throw new RuntimeException("未找到场景ID为 " + sceneId + " 的Chromosome数据");
        }
        
        try {
            // 根据实体名称映射到Chromosome中的字段名
            String fieldName = mapEntityToField(entityName);
            
            // 使用反射获取字段值
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);
            
            // 获取原始数据
            Object originalData = field.get(chromosome);
            
            // 如果原始数据是List类型，则批量更新列表中的项
            if (originalData instanceof List) {
                List<Object> originalList = (List<Object>) originalData;
                int successCount = 0;
                
                for (Map<String, Object> data : dataList) {
                    try {
                        if (data.containsKey("id")) {
                            String id = data.get("id").toString();
                            
                            // 查找并更新匹配的项
                            for (Object item : originalList) {
                                try {
                                    Field idField = item.getClass().getDeclaredField("id");
                                    idField.setAccessible(true);
                                    Object itemId = idField.get(item);
                                    if (itemId != null && itemId.toString().equals(id)) {
                                        // 找到匹配项，更新它
                                        updateObjectFields(item, data);
                                        successCount++;
                                        break;
                                    }
                                } catch (Exception e) {
                                    // 如果没有id字段或访问失败，尝试使用Id字段
                                    try {
                                        Field idField = item.getClass().getDeclaredField("Id");
                                        idField.setAccessible(true);
                                        Object itemId = idField.get(item);
                                        if (itemId != null && itemId.toString().equals(id)) {
                                            // 找到匹配项，更新它
                                            updateObjectFields(item, data);
                                            successCount++;
                                            break;
                                        }
                                    } catch (Exception ex) {
                                        // 忽略异常，继续下一个
                                    }
                                }
                            }
                        }
                    } catch (Exception e) {
                        // 忽略单个数据的异常，继续处理下一个
                    }
                }
                
                // 保存更新后的Chromosome到文件
                boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);
                return saved ? successCount : 0;
            } else {
                throw new RuntimeException("字段 " + entityName + " 不是列表类型，无法进行批量操作");
            }
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + entityName, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 删除Chromosome中的数据
     * @param sceneId 场景ID
     * @param entityName 实体名称
     * @param ids 要删除的数据ID列表
     * @return 删除成功的数量
     */
    public int deleteChromosomeData(String sceneId, String entityName, List<Object> ids) {
        // 从文件中加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            throw new RuntimeException("未找到场景ID为 " + sceneId + " 的Chromosome数据");
        }
        
        try {
            // 根据实体名称映射到Chromosome中的字段名
            String fieldName = mapEntityToField(entityName);
            
            // 使用反射获取字段值
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);
            
            // 获取原始数据
            Object originalData = field.get(chromosome);
            
            // 如果原始数据是List类型，则从列表中删除指定项
            if (originalData instanceof List) {
                List<Object> originalList = (List<Object>) originalData;
                int deleteCount = 0;
                
                // 创建ID字符串列表便于比较
                List<String> idStrings = ids.stream()
                        .map(Object::toString)
                        .collect(Collectors.toList());
                
                // 使用迭代器安全地删除元素
                java.util.Iterator<Object> iterator = originalList.iterator();
                while (iterator.hasNext()) {
                    Object item = iterator.next();
                    try {
                        Field idField = item.getClass().getDeclaredField("id");
                        idField.setAccessible(true);
                        Object itemId = idField.get(item);
                        if (itemId != null && idStrings.contains(itemId.toString())) {
                            // 找到匹配项，删除它
                            iterator.remove();
                            deleteCount++;
                        }
                    } catch (Exception e) {
                        // 如果没有id字段或访问失败，尝试使用Id字段
                        try {
                            Field idField = item.getClass().getDeclaredField("Id");
                            idField.setAccessible(true);
                            Object itemId = idField.get(item);
                            if (itemId != null && idStrings.contains(itemId.toString())) {
                                // 找到匹配项，删除它
                                iterator.remove();
                                deleteCount++;
                            }
                        } catch (Exception ex) {
                            // 忽略异常，继续下一个
                        }
                    }
                }
                
                // 保存更新后的Chromosome到文件
                boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);
                return saved ? deleteCount : 0;
            } else {
                throw new RuntimeException("字段 " + entityName + " 不是列表类型，无法进行删除操作");
            }
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + entityName, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 更新对象的字段值
     * @param obj 要更新的对象
     * @param data 新的字段值
     */
    private void updateObjectFields(Object obj, Map<String, Object> data) {
        Class<?> clazz = obj.getClass();
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            String fieldName = entry.getKey();
            Object fieldValue = entry.getValue();
            
            try {
                Field field = clazz.getDeclaredField(fieldName);
                field.setAccessible(true);
                field.set(obj, fieldValue);
            } catch (Exception e) {
                // 忽略无法设置的字段
            }
        }
    }
}