package com.aps.demo;

import com.aps.entity.Algorithm.OrderSortRule;
import com.aps.entity.basic.Order;
import com.aps.service.Algorithm.OrderSortService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.junit.jupiter.MockitoExtension;

import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

/**
 * 订单排序服务测试类
 */
@ExtendWith(MockitoExtension.class)
class OrderSortServiceTest {

    @InjectMocks
    private OrderSortService orderSortService;

    private List<Order> testOrders;

    @BeforeEach
    void setUp() {
        // 初始化服务（触发@PostConstruct）
        orderSortService.initializeFieldExtractors();
        
        // 创建测试数据
        testOrders = createTestOrders();
    }



    @Test
    void testAssignPriority_WithMultipleConditions() {
        // 创建多条件排序规则：先按dueDate，再按priority
        OrderSortRule rule = createMultiConditionRule();
        
        // 执行测试
        orderSortService.assignPriority(testOrders, rule);

        // 按照优先级排序
        testOrders.sort(
                Comparator.comparing(Order::getActualPriority).reversed()
        );
        
        // 验证多级优先级分配
        printOrderPriorities(testOrders); // 调试输出

    }



    /**
     * 创建测试订单数据
     */
    private List<Order> createTestOrders() {
        List<Order> orders = new ArrayList<>();
        
        // 创建5个测试订单，具有不同的属性
        for (int i = 1; i <=15; i++) {
            Order order = new Order();
            order.setId(i);
            order.setProductId(100 + i);
            order.setQuantity(50.0 * i);
            order.setPriority(15 - i); // 优先级：5,4,3,2,1（倒序）
            //order.setDueDate(OffsetDateTime.now().plusDays(i)); // 到期日递增
            order.setTardiness(i * 0.5);
            order.setCanSplit(i % 2 == 0);
            order.setCanInterrupt(i % 3 == 0);
            
            orders.add(order);
        }
        
        return orders;
    }



    /**
     * 创建多条件排序规则
     */
    private OrderSortRule createMultiConditionRule() {
        OrderSortRule rule = new OrderSortRule();
        rule.setEnabled(true);
        
        List<OrderSortRule.SortCondition> conditions = new ArrayList<>();
        
        // 条件1：按是否可拆分
        OrderSortRule.SortCondition condition1 = new OrderSortRule.SortCondition();
        condition1.setSequence(1);
        condition1.setFieldName("canSplit");
        condition1.setReverse(true); // 可拆分的在前
        conditions.add(condition1);
        
        // 条件2：按到期日
        OrderSortRule.SortCondition condition2 = new OrderSortRule.SortCondition();
        condition2.setSequence(2);
        condition2.setFieldName("dueDate");
        condition2.setReverse(false); // 早到期在前
        conditions.add(condition2);
        
        // 条件3：按优先级
        OrderSortRule.SortCondition condition3 = new OrderSortRule.SortCondition();
        condition3.setSequence(3);
        condition3.setFieldName("priority");
        condition3.setReverse(true); // 高优先级在前
        conditions.add(condition3);
        
        rule.setConditions(conditions);
        return rule;
    }

    /**
     * 打印订单优先级（用于调试）
     */
    private void printOrderPriorities(List<Order> orders) {
        System.out.println("=== 订单优先级分配结果 ===");
        for (Order order : orders) {
            System.out.printf("订单ID: %d, 优先级: %.4f, 到期日: %s, 原优先级: %d, 可拆分: %s%n",
                    order.getId(), order.getActualPriority(), 
                    order.getDueDate(), order.getPriority(), order.isCanSplit());
        }
        System.out.println("========================");
    }
}