package com.aps.service.plan;

import com.aps.entity.Algorithm.Chromosome;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;

/**
 * SceneService.java
 *
 * @author jdt
 * @description 场景保存修改
 * @since 2025-12-01
 */
@Service
public class SceneService {

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(SceneService.class);


    // 统一配置 ObjectMapper
    private ObjectMapper createObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(new JavaTimeModule());
        objectMapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
        // 忽略未知属性，防止反序列化时出错
        objectMapper.configure(com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return objectMapper;
    }

    /**
     * 获取文件存储目录
     */
    private File getResultDirectory() {
        File resultDir = new File("result");
        if (!resultDir.exists()) {
            boolean created = resultDir.mkdirs();
            if (!created) {
                logger.warn("无法创建结果目录: {}", resultDir.getAbsolutePath());
                throw new RuntimeException("无法创建结果目录: " + resultDir.getAbsolutePath());
            }
        }
        return resultDir;
    }

    /**
     * 获取文件路径
     */
    private File getChromosomeFile(String sceneId) {
        File resultDir = getResultDirectory();
        String fileName = "chromosome_result_" + sceneId + ".json";
        return new File(resultDir, fileName);
    }

    /**
     * 从文件中读取Chromosome对象
     */
    public Chromosome loadChromosomeFromFile(String sceneId) {
        if (sceneId == null || sceneId.trim().isEmpty()) {
            logger.warn("场景ID不能为空");
            return null;
        }

        try {
            ObjectMapper objectMapper = createObjectMapper();
            File file = getChromosomeFile(sceneId);

            if (!file.exists()) {
                logger.warn("染色体文件不存在: {}", file.getAbsolutePath());
                return null;
            }

            if (file.length() == 0) {
                logger.warn("染色体文件为空: {}", file.getAbsolutePath());
                return null;
            }

            logger.info("正在从文件加载染色体: {}", file.getAbsolutePath());
            Chromosome chromosome = objectMapper.readValue(file, Chromosome.class);
            logger.info("染色体加载成功，场景ID: {}", sceneId);

            return chromosome;

        } catch (Exception e) {
            // 正确的错误日志写法
            logger.error("加载染色体文件失败，场景ID: " + sceneId, e);
            throw new RuntimeException("加载染色体文件失败: " + e.getMessage(), e);
        }
    }

    /**
     * 将染色体结果保存到JSON文件
     */
    public boolean saveChromosomeToFile(Chromosome chromosome, String sceneId) {
        if (chromosome == null) {
            logger.warn("染色体对象不能为空");
            return false;
        }

        if (sceneId == null || sceneId.trim().isEmpty()) {
            logger.warn("场景ID不能为空");
            return false;
        }

        try {
            ObjectMapper objectMapper = createObjectMapper();
            File file = getChromosomeFile(sceneId);
            File tempFile = new File(file.getParentFile(), file.getName() + ".tmp");

            objectMapper.writeValue(tempFile, chromosome);

            if (tempFile.length() == 0) {
                logger.error("写入的临时文件为空: {}", tempFile.getAbsolutePath());
                Files.deleteIfExists(tempFile.toPath());
                return false;
            }

            // 验证文件内容
            try {
                Chromosome verifyChromosome = objectMapper.readValue(tempFile, Chromosome.class);
                if (verifyChromosome == null) {
                    throw new IOException("验证读取失败");
                }
            } catch (Exception e) {
                logger.error("验证染色体文件内容失败，文件: {}", tempFile.getAbsolutePath(), e);
                Files.deleteIfExists(tempFile.toPath());
                return false;
            }

            // 替换原文件
            if (file.exists()) {
                Files.deleteIfExists(file.toPath());
            }
            Files.move(tempFile.toPath(), file.toPath());

            logger.info("染色体保存成功，场景ID: {}, 文件: {}", sceneId, file.getAbsolutePath());
            return true;

        } catch (Exception e) {
            logger.error("保存染色体文件失败，场景ID: " + sceneId, e);
            return false;
        }
    }

    /**
     * 检查染色体文件是否存在
     */
    public boolean chromosomeFileExists(String sceneId) {
        if (sceneId == null || sceneId.trim().isEmpty()) {
            return false;
        }
        File file = getChromosomeFile(sceneId);
        return file.exists() && file.length() > 0;
    }

    /**
     * 删除染色体文件
     */
    public boolean deleteChromosomeFile(String sceneId) {
        try {
            File file = getChromosomeFile(sceneId);
            if (file.exists()) {
                boolean deleted = Files.deleteIfExists(file.toPath());
                if (deleted) {
                    logger.info("删除染色体文件成功: {}", file.getAbsolutePath());
                }
                return deleted;
            }
            return true;
        } catch (Exception e) {
            logger.error("删除染色体文件失败，场景ID: " + sceneId, e);
            return false;
        }
    }




}
