package com.aps.config;

import io.swagger.v3.oas.models.media.MapSchema;
import io.swagger.v3.oas.models.media.Schema;
import io.swagger.v3.oas.models.media.StringSchema;
import io.swagger.v3.oas.models.examples.Example; // 添加这个导入
import org.springdoc.core.customizers.OperationCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.method.HandlerMethod;

import java.util.HashMap;
import java.util.Map;

@Configuration
public class SwaggerMapParamConfig {

    @Bean
    public OperationCustomizer customizeOperation() {
        return (operation, handlerMethod) -> {
            if (operation.getRequestBody() != null &&
                    operation.getRequestBody().getContent() != null &&
                    operation.getRequestBody().getContent().get("application/json") != null) {

                Schema<?> schema = operation.getRequestBody().getContent().get("application/json").getSchema();
                if (schema instanceof MapSchema) {
                    addParamExamples(operation, handlerMethod);
                }
            }
            return operation;
        };
    }

    private void addParamExamples(io.swagger.v3.oas.models.Operation operation, HandlerMethod handlerMethod) {
        String methodName = handlerMethod.getMethod().getName();
        Map<String, Schema> properties = new HashMap<>();
        Map<String, Example> examples = new HashMap<>(); // 修改这里：Object -> Example

        switch (methodName) {
            case "lanuch":
                properties.put("sceneName", new StringSchema().description("场景名称").example("scene_001"));
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("启动工单示例", createExample(
                        "启动指定场景的工单处理",
                        "{\n" +
                                "    \"sceneName\": \"scene_001\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;

            case "schedule":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("排程计算示例", createExample(
                        "执行场景的排程算法计算",
                        "{\n" +
                                "    \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\"\n" +
                                "}"
                ));
                break;

            case "copyScene":
                properties.put("oldSceneId", new StringSchema().description("原场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("newSceneId", new StringSchema().description("新场景名称").example("scene_002"));
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("完整复制示例", createExample(
                        "复制场景包含所有数据和配置",
                        "{\n" +
                                "    \"oldSceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "    \"newSceneName\": \"scene_002\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                examples.put("最小复制示例", createExample(
                        "仅提供必需参数的场景复制",
                        "{\n" +
                                "    \"oldSceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "    \"newSceneName\": \"scene_002\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;

            case "exportPlan":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("导出计划示例", createExample(
                        "导出指定场景的排程结果为生产计划",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                break;

            case "deleteScene":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("删除场景示例", createExample(
                        "删除指定的场景",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                examples.put("批量删除示例", createExample(
                        "删除多个场景（如果支持批量删除）",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                break;

            case "getAllScene":
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("获取场景示例", createExample(
                        "获取指定用户的所有场景",
                        "{\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                examples.put("管理员查询示例", createExample(
                        "管理员查询系统所有场景（如果支持）",
                        "{\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;


            case "getResourceGantt":
            case "getProductGantt":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                examples.put("示例", createExample(
                        "请求示例",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\"\n" +
                                "}"
                ));
                break;

            case "operationMove":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("id", new StringSchema().description("操作ID").example(1));
                properties.put("newStartTime", new StringSchema().description("新的开始时间").example("2025-11-03T07:36:00.000Z"));
                properties.put("newMachineId", new StringSchema().description("新机器ID").example(3402));
                examples.put("移动操作示例", createExample(
                        "将指定操作移动到新的时间和机器上",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"id\": 1,\n" +
                                "  \"newStartTime\": \"2025-11-03T07:36:00.000Z\",\n" +
                                "  \"newMachineId\": 3402\n" +
                                "}"
                ));
                break;

            case "operationBlockMove":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("id", new StringSchema().description("操作ID列表").example("[1, 2, 3]"));
                properties.put("newStartTime", new StringSchema().description("新的开始时间").example("2025-11-03T07:36:00.000Z"));
                properties.put("newMachineId", new StringSchema().description("新机器ID").example(3402));
                examples.put("移动工作块示例", createExample(
                        "将指定的多个操作移动到新的时间和机器上",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"id\": [1, 2, 3],\n" +
                                "  \"newStartTime\": \"2025-11-03T07:36:00.000Z\",\n" +
                                "  \"newMachineId\": 3401\n" +
                                "}"
                ));
                break;

            case "operationedit":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("operation", new StringSchema().description("操作对象"));
                examples.put("编辑操作示例", createExample(
                        "编辑指定的操作",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"operation\": {}\n" +
                                "}"
                ));
                break;

            case "changeBaseTime":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("baseTime", new StringSchema().description("基础时间").example("2025-11-03T07:36:00.000Z"));
                examples.put("更改基准时间示例", createExample(
                        "更改场景的基准时间",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"baseTime\": \"2025-11-03T07:36:00.000Z\"\n" +
                                "}"
                ));
                break;

            case "fixStartDate":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("id", new StringSchema().description("操作ID").example(1));
                properties.put("newStartTime", new StringSchema().description("新的开始时间").example("2025-11-03T07:36:00.000Z"));
                examples.put("固定开始日期示例", createExample(
                        "固定操作的开始日期",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"id\": 1,\n" +
                                "  \"newStartTime\": \"2025-11-03T07:36:00.000Z\"\n" +
                                "}"
                ));
                break;

            case "spiltOperation":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("id", new StringSchema().description("操作ID").example(1));
                properties.put("splitCounts", new StringSchema().description("拆分数量数组").example("[20000.0, 30000.0]"));
                examples.put("拆分操作示例", createExample(
                        "按指定数量拆分操作",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"id\": 2,\n" +
                                "  \"splitCounts\": [20000, 30000]\n" +
                                "}"
                ));
                break;

            case "delOperation":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("id", new StringSchema().description("操作ID").example(1));
                examples.put("删除操作示例", createExample(
                        "删除指定操作",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"id\": 1\n" +
                                "}"
                ));
                break;

            case "lockedOperation":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("id", new StringSchema().description("操作ID").example(1));
                properties.put("locked", new StringSchema().description("是否锁定").example(true));
                examples.put("锁定操作示例", createExample(
                        "锁定或解锁指定操作",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"id\": 1,\n" +
                                "  \"locked\": true\n" +
                                "}"
                ));
                break;

            case "spiltOrder":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("orderid", new StringSchema().description("订单ID").example("fcc0892a-0483-4da7-8414-9ce98be36e53"));
                properties.put("splitCounts", new StringSchema().description("拆分数量数组").example("[20000.0, 20000.0, 10000.0]"));
                examples.put("拆分订单示例", createExample(
                        "按指定数量拆分订单",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"orderid\": \"fcc0892a-0483-4da7-8414-9ce98be36e53\",\n" +
                                "  \"splitCounts\": [20000, 20000, 10000]\n" +
                                "}"
                ));
                break;

            case "copyOrder":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("orderId", new StringSchema().description("订单ID").example("ORDER001"));
                properties.put("copyCount", new StringSchema().description("复制数量").example(100.0));
                examples.put("订单复制示例", createExample(
                        "复制指定订单",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"orderId\": \"ORDER001\",\n" +
                                "  \"copyCount\": 100.0\n" +
                                "}"
                ));
                break;

            case "delOrder":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("orderId", new StringSchema().description("订单ID").example("ORDER001"));
                examples.put("删除订单示例", createExample(
                        "删除指定订单",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"orderId\": \"ORDER001\"\n" +
                                "}"
                ));
                break;

            case "orderMerge":
                properties.put("sceneId", new StringSchema().description("场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("sourceorderid", new StringSchema().description("源订单ID").example("5be078b725b34ade8b6638f74dad6b10"));
                properties.put("targetorderid", new StringSchema().description("目标订单ID").example("2a0f23d2429f4e5da7b3929da75a803d"));
                examples.put("合并订单示例", createExample(
                        "将源订单合并到目标订单",
                        "{\n" +
                                "  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "  \"sourceorderid\": \"5be078b725b34ade8b6638f74dad6b10\",\n" +
                                "  \"targetorderid\": \"2a0f23d2429f4e5da7b3929da75a803d\"\n" +
                                "}"
                ));
                break;

            // 添加 ChromosomeDataController 相关方法的示例
            case "queryChromosomeData":
                properties.put("sceneId", new StringSchema().description("场景ID").example("SCENE001"));
                properties.put("entityName", new StringSchema().description("实体名称").example("order"));
                properties.put("page", new StringSchema().description("页码").example(1));
                properties.put("size", new StringSchema().description("每页大小").example(10));
                examples.put("订单分页查询示例", createExample(
                        "分页查询订单数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"entityName\": \"order\",\n" +
                                "  \"page\": 1,\n" +
                                "  \"size\": 10\n" +
                                "}"
                ));
                examples.put("设备分页查询示例", createExample(
                        "分页查询设备数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"entityName\": \"machine\",\n" +
                                "  \"page\": 1,\n" +
                                "  \"size\": 10\n" +
                                "}"
                ));
                break;

            case "queryChromosomeDataWithConditions":
                properties.put("entityName", new StringSchema().description("实体名称").example("order"));
                examples.put("订单条件查询示例", createExample(
                        "带条件分页查询订单数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"pageIndex\": 1,\n" +
                                "  \"pageSize\": 10,\n" +
                                "  \"conditions\": [\n" +
                                "    {\n" +
                                "      \"fieldName\": \"id\",\n" +
                                "      \"fieldValue\": \"123\",\n" +
                                "      \"conditionalType\": \"Equal\"\n" +
                                "    }\n" +
                                "  ]\n" +
                                "}"
                ));
                break;

            case "queryChromosomeDataList":
                properties.put("entityName", new StringSchema().description("实体名称").example("order"));
                examples.put("订单列表查询示例", createExample(
                        "查询订单列表数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"conditions\": [\n" +
                                "    {\n" +
                                "      \"fieldName\": \"id\",\n" +
                                "      \"fieldValue\": \"123\",\n" +
                                "      \"conditionalType\": \"Equal\"\n" +
                                "    }\n" +
                                "  ]\n" +
                                "}"
                ));
                break;

            case "queryChromosomeDataById":
                properties.put("sceneId", new StringSchema().description("场景ID").example("SCENE001"));
                properties.put("entityName", new StringSchema().description("实体名称").example("order"));
                properties.put("id", new StringSchema().description("数据ID").example("123"));
                examples.put("订单ID查询示例", createExample(
                        "根据ID查询订单数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"entityName\": \"order\",\n" +
                                "  \"id\": \"123\"\n" +
                                "}"
                ));
                break;

            case "updateChromosomeData":
                properties.put("entityName", new StringSchema().description("实体名称").example("order"));
                examples.put("订单更新示例", createExample(
                        "更新订单数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"data\": {\n" +
                                "    \"id\": \"123\",\n" +
                                "    \"name\": \"updated name\"\n" +
                                "  }\n" +
                                "}"
                ));
                break;

            case "batchChromosomeData":
                properties.put("entityName", new StringSchema().description("实体名称").example("order"));
                examples.put("订单批量操作示例", createExample(
                        "批量操作订单数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"data\": [\n" +
                                "    {\n" +
                                "      \"id\": \"1\",\n" +
                                "      \"name\": \"item1\"\n" +
                                "    },\n" +
                                "    {\n" +
                                "      \"id\": \"2\",\n" +
                                "      \"name\": \"item2\"\n" +
                                "    }\n" +
                                "  ]\n" +
                                "}"
                ));
                break;

            case "deleteChromosomeData":
                properties.put("entityName", new StringSchema().description("实体名称").example("order"));
                examples.put("订单删除示例", createExample(
                        "删除订单数据",
                        "{\n" +
                                "  \"sceneId\": \"SCENE001\",\n" +
                                "  \"ids\": [\"1\", \"2\", \"3\"]\n" +
                                "}"
                ));
                break;
            
            case "changeTimeConfig":
                properties.put("configDate", new StringSchema().description("配置日期").example("2025-12-22T10:00:00"));
                properties.put("freezeHours", new StringSchema().description("冻结小时数").example("2.5"));
                properties.put("preUnitValue", new StringSchema().description("前置单位值").example("1.0"));
                properties.put("postUnitValue", new StringSchema().description("后置单位值").example("1.5"));
                examples.put("完整配置示例", createExample(
                        "完整更新时间配置",
                        "{\n" +
                                "  \"configDate\": \"2025-12-22T10:00:00\",\n" +
                                "  \"freezeHours\": \"2.5\",\n" +
                                "  \"preUnitValue\": \"1.0\",\n" +
                                "  \"postUnitValue\": \"1.5\"\n" +
                                "}"
                ));
                examples.put("部分配置更新示例", createExample(
                        "仅更新部分配置项",
                        "{\n" +
                                "  \"freezeHours\": \"3.0\",\n" +
                                "  \"preUnitValue\": \"2.0\"\n" +
                                "}"
                ));
                break;
            
            case "updateConfig":
                properties.put("baseTime", new StringSchema().description("基准时间").example("2025-12-23T10:00:00"));
                properties.put("freezeDate", new StringSchema().description("冻结日期").example("30"));
                properties.put("startCount", new StringSchema().description("开始计数").example("1"));
                properties.put("endCount", new StringSchema().description("结束计数").example("100"));
                examples.put("APS时间配置更新示例", createExample(
                        "更新APS时间配置",
                        "{\n" +
                                "  \"baseTime\": \"2025-12-23T10:00:00\",\n" +
                                "  \"freezeDate\": 30,\n" +
                                "  \"startCount\": 1,\n" +
                                "  \"endCount\": 100\n" +
                                "}"
                ));
                break;
            
            case "revert":
                properties.put("sceneId", new StringSchema().description("场景ID").example("FAADE046165147629931D5EA6A9015E8"));
                examples.put("回退版本示例", createExample(
                        "回退到上一个版本",
                        "{\n" +
                                "  \"sceneId\": \"FAADE046165147629931D5EA6A9015E8\"\n" +
                                "}"
                ));
                break;
        }

        if (!properties.isEmpty()) {
            MapSchema mapSchema = new MapSchema();
            mapSchema.setProperties(properties);
            mapSchema.setDescription(getMethodDescription(methodName));

            if (!examples.isEmpty()) {
                mapSchema.setExample(examples.values().iterator().next());
                operation.getRequestBody().getContent().get("application/json").setExamples(examples); // 这行现在类型匹配了
            }

            operation.getRequestBody().getContent().get("application/json").setSchema(mapSchema);
        }
    }

    // 修改这里：返回类型从 Map<String, Object> 改为 Example
    private Example createExample(String summary, String value) {
        Example example = new Example();
        example.setSummary(summary);
        example.setValue(value);
        return example;
    }

    private String getMethodDescription(String methodName) {
        switch (methodName) {
            case "lanuch":
                return "启动工单请求参数";
            case "schedule":
                return "排程计算请求参数";
            case "copyScene":
                return "复制场景请求参数";
            case "exportPlan":
                return "导出计划请求参数";
            case "deleteScene":
                return "删除场景请求参数";
            case "getAllScene":
                return "获取场景列表请求参数";

            case "getResourceGantt":

            case "getProductGantt":
                return "获取资源甘特图请求参数";
            case "operationMove":
                return "移动操作请求参数";
            case "operationedit":
                return "编辑操作请求参数";
            case "changebasetime":
                return "更改基准时间请求参数";
            case "spiltOperation":
                return "拆分操作请求参数";
            case "delOperation":
                return "删除操作请求参数";
            case "lockedOperation":
                return "锁定操作请求参数";
            case "spiltOrder":
                return "拆分订单请求参数";
            case "copyOrder":
                return "复制订单请求参数";
            case "delOrder":
                return "删除订单请求参数";
            case "orderMerge":
                return "合并订单请求参数";
            case "fixStartDate":
                return "固定开始日期请求参数";
            // 添加 ChromosomeDataController 相关方法的描述
            case "queryChromosomeData":
                return "分页查询染色体数据请求参数";
            case "queryChromosomeDataWithConditions":
                return "条件分页查询染色体数据请求参数";
            case "queryChromosomeDataList":
                return "条件查询染色体列表数据请求参数";
            case "queryChromosomeDataById":
                return "根据ID查询染色体数据请求参数";
            case "updateChromosomeData":
                return "更新染色体数据请求参数";
            case "batchChromosomeData":
                return "批量操作染色体数据请求参数";
            case "deleteChromosomeData":
                return "删除染色体数据请求参数";
            case "changeTimeConfig":
                return "修改时间配置请求参数";
            case "updateConfig":
                return "更新APS时间配置请求参数";
            case "revert":
                return "回退版本请求参数";
            default:
                return "请求参数";
        }
    }
}