package com.aps.controller;

import com.aps.common.util.R;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.ProdSceneConfig;
import com.aps.service.LanuchService;
import com.aps.service.ProdSceneConfigService;
import com.aps.service.plan.PlanResultService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/lanuch")
@Tag(name = "启动工单", description = "启动工单接口")
public class LanuchController {

    @Autowired
    private LanuchService lanuchService;

    @Autowired
    private PlanResultService planResultService;

    @Autowired
    private ProdSceneConfigService prodSceneConfigService;

    /**
     * 启动工单
     */
    @PostMapping("/execute")
    @Operation(summary = "启动工单")
    public R<String> lanuch(@RequestBody Map<String, String> params) {
        String sceneName = params.get("sceneName");
        String userId = params.get("userId");

        if (sceneName == null || sceneName.trim().isEmpty()) {
            return R.failed("场景名称不能为空");
        }

        return lanuchService.lanuch(sceneName, userId);
    }

    /**
     * 执行排程
     */
    @PostMapping("/schedule")
    @Operation(summary = "运算")
    public R<Chromosome> schedule(@RequestBody Map<String, String> params) {
        String sceneId = params.get("sceneId");

        if (sceneId == null || sceneId.trim().isEmpty()) {
            return R.failed("场景ID不能为空");
        }

        try {
            Chromosome scheduleChromosomes = planResultService.schedule(sceneId);
            return R.ok(scheduleChromosomes);
        } catch (Exception e) {
            return R.failed("排程执行失败: " + e.getMessage());
        }
    }

    /**
     * 复制场景
     */
    @PostMapping("/copyScene")
    @Operation(summary = "复制场景")
    public R<String> copyScene(@RequestBody Map<String, String> params) {
        String oldSceneId = params.get("oldSceneId");
        String newSceneName = params.get("newSceneName");
        String userId = params.get("userId");

        if (oldSceneId == null || oldSceneId.trim().isEmpty()) {
            return R.failed("原场景ID不能为空");
        }
        if (newSceneName == null || newSceneName.trim().isEmpty()) {
            return R.failed("新场景名称不能为空");
        }

        if (userId == null || userId.trim().isEmpty()) {
            return R.failed("用户名不能为空");
        }
        return lanuchService.copyScene(oldSceneId, userId, newSceneName);
    }

    /**
     * 导出计划
     */
    @PostMapping("/exportPlan")
    @Operation(summary = "导出计划")
    public R<String> exportPlan(@RequestBody Map<String, String> params) {
        String sceneId = params.get("sceneId");

        if (sceneId == null || sceneId.trim().isEmpty()) {
            return R.failed("场景ID不能为空");
        }

        return lanuchService.exportPlan(sceneId);
    }

    /**
     * 删除场景
     */
    @PostMapping("/deleteScene")
    @Operation(summary = "删除场景")
    public R<Boolean> deleteScene(@RequestBody Map<String, String> params) {
        String sceneId = params.get("sceneId");

        if (sceneId == null || sceneId.trim().isEmpty()) {
            return R.failed("场景ID不能为空");
        }

        boolean result = prodSceneConfigService.deleteSceneById(sceneId);
        return result ? R.ok(true) : R.failed("删除场景失败");
    }

    /**
     * 获取用户所有场景
     */
    @PostMapping("/getAllScene")
    @Operation(summary = "获取用户场景")
    public R<List<ProdSceneConfig>> getAllScene(@RequestBody Map<String, String> params) {
        String userId = params.get("userId");

        if (userId == null || userId.trim().isEmpty()) {
            return R.failed("用户ID不能为空");
        }

        List<ProdSceneConfig> scenes = prodSceneConfigService.lambdaQuery()
                .eq(ProdSceneConfig::getCreateUser, userId)
                .list();
        return R.ok(scenes);
    }
}