package com.aps.config;

import io.swagger.v3.oas.models.media.MapSchema;
import io.swagger.v3.oas.models.media.Schema;
import io.swagger.v3.oas.models.media.StringSchema;
import io.swagger.v3.oas.models.examples.Example; // 添加这个导入
import org.springdoc.core.customizers.OperationCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.method.HandlerMethod;

import java.util.HashMap;
import java.util.Map;

@Configuration
public class SwaggerMapParamConfig {

    @Bean
    public OperationCustomizer customizeOperation() {
        return (operation, handlerMethod) -> {
            if (operation.getRequestBody() != null &&
                    operation.getRequestBody().getContent() != null &&
                    operation.getRequestBody().getContent().get("application/json") != null) {

                Schema<?> schema = operation.getRequestBody().getContent().get("application/json").getSchema();
                if (schema instanceof MapSchema) {
                    addParamExamples(operation, handlerMethod);
                }
            }
            return operation;
        };
    }

    private void addParamExamples(io.swagger.v3.oas.models.Operation operation, HandlerMethod handlerMethod) {
        String methodName = handlerMethod.getMethod().getName();
        Map<String, Schema> properties = new HashMap<>();
        Map<String, Example> examples = new HashMap<>(); // 修改这里：Object -> Example

        switch (methodName) {
            case "lanuch":
                properties.put("sceneName", new StringSchema().description("场景名称").example("scene_001"));
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("启动工单示例", createExample(
                        "启动指定场景的工单处理",
                        "{\n" +
                                "    \"sceneName\": \"scene_001\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;

            case "schedule":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("排程计算示例", createExample(
                        "执行场景的排程算法计算",
                        "{\n" +
                                "    \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\"\n" +
                                "}"
                ));
                break;

            case "copyScene":
                properties.put("oldSceneId", new StringSchema().description("原场景ID").example("B571EF6682DB463AB2977B1055A74112"));
                properties.put("newSceneId", new StringSchema().description("新场景名称").example("scene_002"));
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("完整复制示例", createExample(
                        "复制场景包含所有数据和配置",
                        "{\n" +
                                "    \"oldSceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "    \"newSceneName\": \"scene_002\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                examples.put("最小复制示例", createExample(
                        "仅提供必需参数的场景复制",
                        "{\n" +
                                "    \"oldSceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n" +
                                "    \"newSceneName\": \"scene_002\",\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;

            case "exportPlan":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("导出计划示例", createExample(
                        "导出指定场景的排程结果为生产计划",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                break;

            case "deleteScene":
                properties.put("sceneId", new StringSchema().description("场景ID").example("487282ECAF57435F9373BAAAFBE90616"));
                examples.put("删除场景示例", createExample(
                        "删除指定的场景",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                examples.put("批量删除示例", createExample(
                        "删除多个场景（如果支持批量删除）",
                        "{\n" +
                                "    \"sceneId\": \"487282ECAF57435F9373BAAAFBE90616\"\n" +
                                "}"
                ));
                break;

            case "getAllScene":
                properties.put("userId", new StringSchema().description("用户ID").example("111"));
                examples.put("获取场景示例", createExample(
                        "获取指定用户的所有场景",
                        "{\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                examples.put("管理员查询示例", createExample(
                        "管理员查询系统所有场景（如果支持）",
                        "{\n" +
                                "    \"userId\": \"111\"\n" +
                                "}"
                ));
                break;
        }

        if (!properties.isEmpty()) {
            MapSchema mapSchema = new MapSchema();
            mapSchema.setProperties(properties);
            mapSchema.setDescription(getMethodDescription(methodName));

            if (!examples.isEmpty()) {
                mapSchema.setExample(examples.values().iterator().next());
                operation.getRequestBody().getContent().get("application/json").setExamples(examples); // 这行现在类型匹配了
            }

            operation.getRequestBody().getContent().get("application/json").setSchema(mapSchema);
        }
    }

    // 修改这里：返回类型从 Map<String, Object> 改为 Example
    private Example createExample(String summary, String value) {
        Example example = new Example();
        example.setSummary(summary);
        example.setValue(value);
        return example;
    }

    private String getMethodDescription(String methodName) {
        switch (methodName) {
            case "lanuch":
                return "启动工单请求参数";
            case "schedule":
                return "排程计算请求参数";
            case "copyScene":
                return "复制场景请求参数";
            case "exportPlan":
                return "导出计划请求参数";
            case "deleteScene":
                return "删除场景请求参数";
            case "getAllScene":
                return "获取场景列表请求参数";
            default:
                return "请求参数";
        }
    }
}
