package com.aps.common.util;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import java.math.BigDecimal;
import java.util.Optional;

@Slf4j
@UtilityClass
public class NumberUtils {
    
    /**
     * 安全地将对象转换为Double
     */
    public static Double safeToDouble(Object obj) {
        return safeToDouble(obj, 0.0);
    }
    
    /**
     * 安全地将对象转换为Double，可指定默认值
     */
    public static Double safeToDouble(Object obj, Double defaultValue) {
        if (obj == null) {
            return defaultValue;
        }
        
        if (obj instanceof Number) {
            return ((Number) obj).doubleValue();
        }
        
        if (obj instanceof String) {
            try {
                String str = ((String) obj).trim();
                return str.isEmpty() ? defaultValue : Double.parseDouble(str);
            } catch (NumberFormatException e) {
                log.warn("字符串转Double失败: {}, 使用默认值: {}", obj, defaultValue);
                return defaultValue;
            }
        }
        
        if (obj instanceof Boolean) {
            return ((Boolean) obj) ? 1.0 : 0.0;
        }
        
        log.warn("不支持的数据类型: {}, 值: {}, 使用默认值: {}", 
                obj.getClass().getSimpleName(), obj, defaultValue);
        return defaultValue;
    }
    
    /**
     * 安全转换为BigDecimal
     */
    public static BigDecimal safeToBigDecimal(Object obj, BigDecimal defaultValue) {
        if (obj == null) {
            return defaultValue;
        }
        
        try {
            if (obj instanceof BigDecimal) {
                return (BigDecimal) obj;
            }
            if (obj instanceof Number) {
                return BigDecimal.valueOf(((Number) obj).doubleValue());
            }
            if (obj instanceof String) {
                String str = ((String) obj).trim();
                return str.isEmpty() ? defaultValue : new BigDecimal(str);
            }
        } catch (Exception e) {
            log.warn("转换BigDecimal失败: {}, 使用默认值", obj, e);
        }
        
        return defaultValue;
    }
    
    /**
     * 检查是否为有效数字
     */
    public static boolean isValidDouble(Double value) {
        return value != null && !value.isNaN() && !value.isInfinite();
    }
}