package com.aps.controller.common;

import com.aps.common.util.R;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.basic.Entry;
import com.aps.entity.basic.MachineOption;
import com.aps.entity.common.Paged;
import com.aps.service.common.ChromosomeDataService;
import com.aps.service.plan.PlanResultService;
import com.aps.service.plan.SceneService;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/queryChromosome")
@Tag(name = "甘特图数据查询", description = "甘特图数据查询")
public class ChromosomeDataController {

    @Autowired
    private ChromosomeDataService chromosomeDataService;
    @Autowired
    private SceneService sceneService;
    @Autowired
    private PlanResultService planResultService;

    /**
     * 通用接口，根据实体名称查询Chromosome中的数据，支持分页和条件过滤
     * 示例:
     * - 文件实体: POST /queryChromosome/order/page
     *   Body: { "sceneId": "SCENE001", "pageIndex": 1, "pageSize": 10, "conditions": [...] }
     * - 数据库实体: POST /queryChromosome/user/page
     *   Body: { "pageIndex": 1, "pageSize": 10, "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine, user, department等)
     * @param paged 分页和条件对象
     * @return 分页数据
     */
    @PostMapping("/{entityName}/page")
    public R<Map<String, Object>> queryChromosomeDataWithConditions(
            @PathVariable String entityName,
            @RequestBody Paged paged) {


            // 获取sceneId（文件实体需要，数据库实体可选）
            String sceneId = paged.getCondition("sceneId") != null ?
                    paged.getCondition("sceneId").getFieldValue() : null;

            // 文件实体必须要有sceneId
            if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
                return R.ok(null);
            }

            Map<String, Object> result = chromosomeDataService.queryChromosomeDataWithConditions(
                    sceneId, entityName, paged);
            return R.ok(result);

    }

    /**
     * 通用接口，根据实体名称查询Chromosome中的列表数据，支持条件过滤
     * 示例:
     * - 文件实体: POST /queryChromosome/order/list
     *   Body: { "sceneId": "SCENE001", "conditions": [...] }
     * - 数据库实体: POST /queryChromosome/user/list
     *   Body: { "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine, user, department等)
     * @param paged 条件对象
     * @return 列表数据
     */
    @PostMapping("/{entityName}/list")
    public R<List<Object>> queryChromosomeDataList(
            @PathVariable String entityName,
            @RequestBody Paged paged) {


        // 获取sceneId（文件实体需要，数据库实体可选）
        String sceneId = paged.getCondition("sceneId") != null ?
                paged.getCondition("sceneId").getFieldValue() : null;

        // 文件实体必须要有sceneId
        if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
            return R.ok(null);
        }

        // 直接调用服务层的list查询方法
        List<Object> result = chromosomeDataService.queryChromosomeDataList(sceneId, entityName, paged);


        if (result == null) {
            return R.ok(Collections.emptyList(), "查询成功，但未找到匹配的数据");
        }

        return R.ok(result);

    }

    /**
     * 通用接口，根据实体名称和ID查询Chromosome中的单个数据
     * 示例:
     * - 文件实体: GET /queryChromosome/order/get?sceneId=xxx&id=123
     * - 数据库实体: GET /queryChromosome/user/get?id=123
     *
     * @param sceneId 场景ID (文件实体必需，数据库实体可选)
     * @param entityName 实体名称
     * @param id 数据ID
     * @return 单个数据对象
     */
    @GetMapping("/{entityName}/get")
    public R<Object> queryChromosomeDataById(
            @RequestParam(required = false) String sceneId,
            @PathVariable String entityName,
            @RequestParam String id) {


            // 文件实体必须要有sceneId
            if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
                return R.failed("文件实体查询时sceneId不能为空");
            }

            Object result = chromosomeDataService.queryChromosomeDataById(sceneId, entityName, id);

            return R.ok(result);



    }

    /**
     * 通用接口，根据实体名称更新Chromosome中的数据
     * 支持文件实体（通过sceneId）和数据库实体的更新
     * 示例:
     * - 文件实体: PUT /queryChromosome/order/update?sceneId=xxx
     *   Body: { "id": 123, "fieldName": "newValue", ... }
     * - 数据库实体: PUT /queryChromosome/user/update
     *   Body: { "id": 123, "fieldName": "newValue", ... }
     *
     * @param sceneId 场景ID (文件实体必需，数据库实体可选)
     * @param entityName 实体名称
     * @param data 要更新的数据 (必须包含id字段)
     * @return 更新结果
     */
    @PostMapping("/{entityName}/update")
    @Operation(summary = "更新实体数据", description = "根据实体名称和ID更新数据，支持文件实体和数据库实体")
    public R<String> updateChromosomeData(
            @Parameter(description = "场景ID (文件实体必需，数据库实体可选)", required = false)
            @RequestParam(required = false) String sceneId,
            @Parameter(description = "taskID (文件实体必需，数据库实体可选)", required = false)
            @RequestParam(required = false) String taskId,
            @Parameter(description = "实体名称", required = true)
            @PathVariable String entityName,
            @Parameter(description = "要更新的数据 (必须包含id字段)", required = true)
            @RequestBody Map<String, Object> data) {

        // 文件实体必须要有sceneId
        if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
            return R.failed("文件实体更新时sceneId不能为空");
        }
        if (entityName.equalsIgnoreCase("machineOption")){


            return updateMachineOption(sceneId, taskId, data);
        }

        try {
            boolean success = chromosomeDataService.updateChromosomeData(sceneId, entityName, data);
            if (success) {
                return R.ok("更新成功");
            } else {
                return R.failed("更新失败");
            }
        } catch (Exception e) {
            return R.failed("更新失败: " + e.getMessage());
        }
    }


    private R<String> updateMachineOption(String sceneId, String taskId, Map<String, Object> data) {
        // 验证taskId不能为空
        if (taskId == null || taskId.isEmpty()) {
            return R.failed("更新machineOption时taskId不能为空");
        }


        try {
            // 1. 加载Chromosome对象
            Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
            if (chromosome == null) {
                return R.failed("未找到场景ID为 " + sceneId + " 的Chromosome数据");
            }

            // 2. 根据taskId找到对应的entry
            Entry targetEntry = null;
            for (Entry entry : chromosome.getAllOperations()) {
                if (String.valueOf(entry.getId()).equals(taskId)) {
                    targetEntry = entry;
                    break;
                }
            }

            if (targetEntry == null) {
                return R.failed("未找到taskId为 " + taskId + " 的entry数据");
            }

            // 3. 获取machineOption的id
            String machineOptionId = String.valueOf(data.get("machineId"));

            // 4. 在entry的machineOptions列表中找到对应的machineOption
            MachineOption targetMachineOption = null;
            for (MachineOption machineOption : targetEntry.getMachineOptions()) {
                if (String.valueOf(machineOption.getMachineId()).equals(machineOptionId)) {
                    targetMachineOption = machineOption;
                    break;
                }
            }

            if (targetMachineOption == null) {
                return R.failed("未找到id为 " + machineOptionId + " 的machineOption数据");
            }

            // 5. 更新machineOption的字段
            ObjectMapper objectMapper = new ObjectMapper();
            MachineOption updatedMachineOption = objectMapper.convertValue(data, MachineOption.class);

            // 6. 替换原来的machineOption
            int index = targetEntry.getMachineOptions().indexOf(targetMachineOption);
            targetEntry.getMachineOptions().set(index, updatedMachineOption);
            planResultService.editMachine(chromosome, sceneId,targetEntry);
            // 7. 保存更新后的Chromosome
            boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);
            if (saved) {
                return R.ok("更新成功");
            } else {
                return R.failed("更新失败");
            }

        } catch (Exception e) {
            return R.failed("更新machineOption失败: " + e.getMessage());
        }

    }



    /**
     * 同一entry下的多个machineOption更新
     */
    private R<String> updateMultipleMachineOptionsInEntry(String sceneId, String taskId, Map<String, Object> data) {
        try {
            // 1. 加载Chromosome对象
            Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
            if (chromosome == null) {
                return R.failed("未找到场景ID为 " + sceneId + " 的Chromosome数据");
            }

            // 2. 根据taskId找到对应的entry
            Entry targetEntry = null;
            for (Entry entry : chromosome.getAllOperations()) {
                if (String.valueOf(entry.getId()).equals(taskId)) {
                    targetEntry = entry;
                    break;
                }
            }

            if (targetEntry == null) {
                return R.failed("未找到taskId为 " + taskId + " 的entry数据");
            }

            // 3. 获取machineOptions列表
            Object machineOptionsObj = data.get("machineOptions");
            if (!(machineOptionsObj instanceof List)) {
                return R.failed("machineOptions必须是数组格式");
            }

            List<?> machineOptionsList = (List<?>) machineOptionsObj;
            if (machineOptionsList.isEmpty()) {
                return R.failed("machineOptions数组不能为空");
            }

            // 4. 更新当前entry下的所有machineOption
            ObjectMapper objectMapper = new ObjectMapper();
            boolean allSuccess = true;
            List<String> errorMessages = new ArrayList<>();

            for (Object optionObj : machineOptionsList) {
                if (!(optionObj instanceof Map)) {
                    allSuccess = false;
                    errorMessages.add("machineOptions中的元素必须是对象格式");
                    continue;
                }

                Map<String, Object> optionMap = (Map<String, Object>) optionObj;

                // 获取machineOption的machineId
                String machineOptionId = String.valueOf(optionMap.get("machineId"));
                if (machineOptionId == null) {
                    allSuccess = false;
                    errorMessages.add("machineOption必须包含machineId字段");
                    continue;
                }

                // 在entry的machineOptions列表中找到对应的machineOption
                MachineOption targetMachineOption = null;
                for (MachineOption machineOption : targetEntry.getMachineOptions()) {
                    if (String.valueOf(machineOption.getMachineId()).equals(machineOptionId)) {
                        targetMachineOption = machineOption;
                        break;
                    }
                }

                if (targetMachineOption == null) {
                    allSuccess = false;
                    errorMessages.add("未找到machineId为 " + machineOptionId + " 的machineOption数据");
                    continue;
                }

                // 更新machineOption的字段
                MachineOption updatedMachineOption = objectMapper.convertValue(optionMap, MachineOption.class);

                // 替换原来的machineOption
                int index = targetEntry.getMachineOptions().indexOf(targetMachineOption);
                targetEntry.getMachineOptions().set(index, updatedMachineOption);
            }

            if (!allSuccess) {
                return R.failed("部分更新失败: " + String.join("; ", errorMessages));
            }

            // 5. 调用editMachine方法处理entry更新
            planResultService.editMachine(chromosome, sceneId, targetEntry);

            // 6. 保存更新后的Chromosome
            boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);
            if (saved) {
                return R.ok("批量更新成功");
            } else {
                return R.failed("批量更新失败: 保存Chromosome数据失败");
            }

        } catch (Exception e) {
            return R.failed("批量更新machineOption失败: " + e.getMessage());
        }

    }
    /**
     * 判断是否为文件实体
     */
    private boolean isFileEntity(String entityName) {
        // 这里列出所有文件实体的名称
        String[] fileEntities = {"order", "entry", "machine", "globaloperationinfo", "groupresult", "prodprocessexec", "machineoption"};
        for (String fileEntity : fileEntities) {
            if (fileEntity.equalsIgnoreCase(entityName)) {
                return true;
            }
        }
        return false;
    }
}