package com.aps.service.Algorithm;

import com.aps.entity.Algorithm.OrderSortRule;
import com.aps.entity.basic.Order;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.lang.reflect.Method;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 增强版订单排序服务（最终版：严格匹配条件数格式）
 */
@Slf4j
@Service
public class OrderSortService {

    // 字段提取器映射（静态常量，所有实例共享）
    private static final Map<String, Function<Order, ?>> FIELD_EXTRACTORS = new HashMap<>();

    /**
     * 初始化字段提取器（使用Spring的PostConstruct）
     */
    @PostConstruct
    public void initializeFieldExtractors() {
        registerFieldExtractor("id", Order::getId);


        registerFieldExtractor("startDate", Order -> {
            // 直接返回LocalDate，处理null情况
            return Optional.ofNullable(Order.getStartDate())
                    .map(LocalDateTime::toLocalDate)
                    .orElse(null);
        });
        registerFieldExtractor("dueDate", Order -> {
            // 直接返回LocalDate，处理null情况
            return Optional.ofNullable(Order.getDueDate())
                    .map(LocalDateTime::toLocalDate)
                    .orElse(null);
        });



        registerFieldExtractor("groupStartDate", Order -> {
            // 直接返回LocalDate，处理null情况
            return Optional.ofNullable(Order.getGroupStartDate())
                    .map(LocalDateTime::toLocalDate)
                    .orElse(null);
        });
        registerFieldExtractor("groupDueDate", Order -> {
            // 直接返回LocalDate，处理null情况
            return Optional.ofNullable(Order.getGroupDueDate())
                    .map(LocalDateTime::toLocalDate)
                    .orElse(null);
        });
        registerFieldExtractor("priority", Order::getPriority);
        registerFieldExtractor("materialCode", Order::getMaterialCode);
        registerFieldExtractor("serie", Order::getSerie);
        registerFieldExtractor("orderCode", Order::getOrderCode);
    }

    /**
     * 注册字段提取器（统一处理字段名小写）
     */
    private void registerFieldExtractor(String fieldName, Function<Order, ?> extractor) {
        FIELD_EXTRACTORS.put(fieldName.toLowerCase(), extractor);
    }


    /**
     * 分配优先级（更新原列表）
     */
    public void assignPriority(List<Order> orders, OrderSortRule rule) {
        Objects.requireNonNull(orders, "Orders list must not be null");
        Objects.requireNonNull(rule, "Sort rule must not be null");

        if (CollectionUtils.isEmpty(orders)) {
            return;
        }

        List<Order> sortedOrders = assignPriorityValues(new ArrayList<>(orders), rule);
        orders.clear();
        orders.addAll(sortedOrders);
    }


    /**
     * 根据数据库配置的规则对订单排序
     */
    public List<Order> sortByRule(List<Order> orders, OrderSortRule rule) {
        Objects.requireNonNull(orders, "Orders list must not be null");
        Objects.requireNonNull(rule, "Sort rule must not be null");

        if (CollectionUtils.isEmpty(orders)) {
            return new ArrayList<>();
        }

        if (rule == null || !rule.isEnabled() || CollectionUtils.isEmpty(rule.getConditions())) {
            return new ArrayList<>(orders);
        }

        List<OrderSortRule.SortCondition> sortedConditions = rule.getConditions().stream()
                .sorted(Comparator.comparingInt(OrderSortRule.SortCondition::getSequence))
                .collect(Collectors.toList());

        Comparator<Order> comparator = null;
        for (OrderSortRule.SortCondition condition : sortedConditions) {
            Comparator<Order> currentComparator = createConditionComparator(condition);
            comparator = comparator == null ? currentComparator : comparator.thenComparing(currentComparator);
        }

        return orders.stream()
                .sorted(comparator != null ? comparator : Comparator.comparing(Order::getId))
                .collect(Collectors.toList());
    }

    /**
     * 分配层级化序号式优先级（严格匹配条件数格式）
     */
    public List<Order> assignPriorityValues(List<Order> orders, OrderSortRule rule) {
        Objects.requireNonNull(orders, "Orders list must not be null");
        Objects.requireNonNull(rule, "Sort rule must not be null");

        if (CollectionUtils.isEmpty(orders)) {
            return new ArrayList<>();
        }

        if (rule == null || !rule.isEnabled() || CollectionUtils.isEmpty(rule.getConditions())) {
            return new ArrayList<>(orders);
        }

        // 按条件顺序排序（sequence越小越先执行）
        List<OrderSortRule.SortCondition> sortedConditions = rule.getConditions().stream()
                .sorted(Comparator.comparingInt(OrderSortRule.SortCondition::getSequence))
                .collect(Collectors.toList());

        // 为每个订单初始化优先级路径
        Map<Order, List<Integer>> priorityPaths = new HashMap<>();
        orders.forEach(order -> priorityPaths.put(order, new ArrayList<>()));

        // 递归分配层级化优先级（核心逻辑）
        assignHierarchicalPriority(orders, sortedConditions, 0, priorityPaths);

        // 将路径转换为最终格式
        convertPriorityPathsToNumeric(orders, priorityPaths);

        // 按优先级升序排序（数值越小越优先）
        return orders.stream()
                .sorted(Comparator.comparingDouble(Order::getActualPriority))
                .collect(Collectors.toList());
    }

    /**
     * 递归分配层级化优先级路径
     */
    private void assignHierarchicalPriority(List<Order> orders,
                                            List<OrderSortRule.SortCondition> conditions,
                                            int conditionIndex,
                                            Map<Order, List<Integer>> priorityPaths) {
        // 递归终止：处理完所有条件后停止，不再添加额外序号
        if (conditionIndex >= conditions.size() || CollectionUtils.isEmpty(orders)) {
            return;
        }

        OrderSortRule.SortCondition currentCondition = conditions.get(conditionIndex);
        String fieldName = currentCondition.getFieldName();

        Function<Order, ?> keyExtractor = getFieldExtractor(fieldName);
        if (keyExtractor == null) {
            log.warn("跳过无效排序字段：{}", currentCondition.getFieldName());
            assignHierarchicalPriority(orders, conditions, conditionIndex + 1, priorityPaths);
            return;
        }

        // 1. 按当前条件分组
        Map<Object, List<Order>> groups = orders.stream()
                .collect(Collectors.groupingBy(keyExtractor));

        // 2. 对分组键排序（关键：按条件配置的方向排序）
        List<Object> sortedKeys = getSortedKeys(groups, currentCondition);

        // 3. 为每个分组分配层级序号（从1开始）
        for (int groupIndex = 0; groupIndex < sortedKeys.size(); groupIndex++) {
            Object key = sortedKeys.get(groupIndex);
            List<Order> groupOrders = groups.get(key);

            // 分配当前层级序号（1、2、3...）
            int levelNumber = groupIndex + 1;
            groupOrders.forEach(order -> priorityPaths.get(order).add(levelNumber));

            // 递归处理下一级条件
            assignHierarchicalPriority(groupOrders, conditions, conditionIndex + 1, priorityPaths);
        }
    }

    /**
     * 路径转数值：严格匹配条件数格式
     * 3个条件 → [1,1,1] → 1.11
     * 4个条件 → [1,1,1,1] → 1.111
     */
    private void convertPriorityPathsToNumeric(List<Order> orders, Map<Order, List<Integer>> priorityPaths) {
        for (Order order : orders) {
            List<Integer> path = priorityPaths.get(order);
            if (CollectionUtils.isEmpty(path)) {
                order.setActualPriority(0.0);
                continue;
            }

            try {
                // 第一部分作为整数
                StringBuilder sb = new StringBuilder();
                sb.append(path.get(0));

                // 后续部分作为小数（严格按照路径长度）
                if (path.size() > 1) {
                    sb.append(".");
                    for (int i = 1; i < path.size(); i++) {
                        sb.append(path.get(i));
                    }
                }

                // 转换为double
                order.setActualPriority(Double.parseDouble(sb.toString()));

            } catch (NumberFormatException e) {
                log.error("优先级格式转换失败：{} → {}", path, e.getMessage());
                order.setActualPriority(0.0);
            }
        }
    }

    /**
     * 创建单个条件的比较器
     */
    @SuppressWarnings("unchecked")
    private Comparator<Order> createConditionComparator(OrderSortRule.SortCondition condition) {
        Objects.requireNonNull(condition, "Sort condition must not be null");

        String fieldName = condition.getFieldName();
        // 不再替换为dueDateOnly，直接使用原字段名

        Function<Order, ?> keyExtractor = getFieldExtractor(fieldName);
        if (keyExtractor == null) {
            log.warn("无字段提取器：{}，使用ID排序", condition.getFieldName());
            return Comparator.comparing(Order::getId);
        }

        Comparator<Object> fieldComparator = getFieldComparator(condition, fieldName);
        return (Comparator<Order>) Comparator.comparing(keyExtractor, fieldComparator);
    }

    /**
     * 获取字段比较器（支持reverse参数）
     */
    @SuppressWarnings("unchecked")
    private Comparator<Object> getFieldComparator(OrderSortRule.SortCondition condition, String actualFieldName) {
        Objects.requireNonNull(condition, "Sort condition must not be null");

        // 自定义比较器优先
        if (condition.getCustomComparator() != null && !condition.getCustomComparator().isEmpty()) {
            try {
                Class<?> clazz = Class.forName(condition.getCustomComparator());
                Method method = clazz.getMethod("getInstance");
                return (Comparator<Object>) method.invoke(null);
            } catch (Exception e) {
                log.error("自定义比较器加载失败：{}", condition.getCustomComparator(), e);
            }
        }

        // 按字段类型返回比较器
        Function<Order, ?> extractor = getFieldExtractor(actualFieldName);
        if (extractor == null) {
            Comparator<Object> defaultComparator = Comparator.comparing(Object::toString);
            return condition.isReverse() ? defaultComparator.reversed() : defaultComparator;
        }

        try {
            Order sample = new Order();
            Object sampleValue = extractor.apply(sample);

            // 布尔类型排序（支持reverse参数）
            if (sampleValue instanceof Boolean) {
                Comparator<Object> booleanComparator = (Comparator<Object>) (o1, o2) -> {
                    boolean b1 = (Boolean) o1;
                    boolean b2 = (Boolean) o2;
                    return Boolean.compare(b1, b2);
                };

                return condition.isReverse() ? booleanComparator.reversed() : booleanComparator;
            }
            // 日期类型：按时间排序
            else if (sampleValue instanceof LocalDate) {
                Comparator<Object> dateComparator = (Comparator<Object>) Comparator.comparing(o -> (LocalDate) o);
                return condition.isReverse() ? dateComparator.reversed() : dateComparator;
            }
            // 整数类型
            else if (sampleValue instanceof Integer) {
                Comparator<Object> intComparator = (Comparator<Object>) Comparator.comparingInt(o -> (Integer) o);
                return condition.isReverse() ? intComparator.reversed() : intComparator;
            }
            // 长整型
            else if (sampleValue instanceof Long) {
                Comparator<Object> longComparator = (Comparator<Object>) Comparator.comparingLong(o -> (Long) o);
                return condition.isReverse() ? longComparator.reversed() : longComparator;
            }
            // 日期时间类型
            else if (sampleValue instanceof OffsetDateTime) {
                Comparator<Object> dateTimeComparator = (Comparator<Object>) Comparator.comparing(o -> (OffsetDateTime) o);
                return condition.isReverse() ? dateTimeComparator.reversed() : dateTimeComparator;
            }
        } catch (Exception e) {
            log.debug("字段类型推断失败：{}", actualFieldName, e);
        }

        // 默认比较器
        Comparator<Object> defaultComparator = Comparator.comparing(Object::toString);
        return condition.isReverse() ? defaultComparator.reversed() : defaultComparator;
    }

    /**
     * 获取排序后的分组键
     */
    private List<Object> getSortedKeys(Map<Object, List<Order>> groups,
                                       OrderSortRule.SortCondition condition) {
        Objects.requireNonNull(groups, "Groups map must not be null");
        Objects.requireNonNull(condition, "Sort condition must not be null");

        String fieldName = condition.getFieldName();

        Comparator<Object> comparator = getFieldComparator(condition, fieldName);
        return groups.keySet().stream()
                .sorted(comparator)
                .collect(Collectors.toList());
    }

    /**
     * 获取字段提取器
     */
    private Function<Order, ?> getFieldExtractor(String fieldName) {
        return fieldName == null ? null : FIELD_EXTRACTORS.get(fieldName.toLowerCase());
    }

    /**
     * 获取格式化的优先级字符串
     */
    public String getFormattedPriority(double actualPriority) {
        return String.valueOf(actualPriority);
    }
}