package com.aps.service.plan;

import com.aps.common.util.redis.RedisUtils;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Schedule.SceneChromsome;
import com.aps.entity.Schedule.SceneDetail;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.List;
import java.util.stream.Collectors;

/**
 * SceneService.java
 *
 * @author jdt
 * @description 场景保存修改
 * @since 2025-12-01
 */
@Service
public class SceneService {

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(SceneService.class);

    @Autowired
    private RedisUtils redisUtils;

    // 统一配置 ObjectMapper
    private ObjectMapper createObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(new JavaTimeModule());
        objectMapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
        // 忽略未知属性，防止反序列化时出错
        objectMapper.configure(com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return objectMapper;
    }

    /**
     * 获取文件存储目录
     */
    private File getResultDirectory() {
        File resultDir = new File("result");
        if (!resultDir.exists()) {
            boolean created = resultDir.mkdirs();
            if (!created) {
                logger.warn("无法创建结果目录: {}", resultDir.getAbsolutePath());
                throw new RuntimeException("无法创建结果目录: " + resultDir.getAbsolutePath());
            }
        }
        return resultDir;
    }

    /**
     * 获取文件路径
     */
    private File getChromosomeFile(String sceneId) {
        File resultDir = getResultDirectory();
        String fileName = "chromosome_result_" + sceneId + ".json";
        return new File(resultDir, fileName);
    }

    private File getChromosomeFile(String sceneId,String ver) {
        File resultDir = getResultDirectory();
        String fileName = "chromosome_result_" + sceneId +"_"+ver+"_.json";
        return new File(resultDir, fileName);
    }

    /**
     * 从文件中读取Chromosome对象
     */
    public Chromosome loadChromosomeFromFile(String sceneId) {
        if (sceneId == null || sceneId.trim().isEmpty()) {
            logger.warn("场景ID不能为空");
            return null;
        }

        try {
            ObjectMapper objectMapper = createObjectMapper();
            SceneChromsome sceneChromsome=(SceneChromsome)redisUtils.get("SceneId."+sceneId);
            File file;
            if(sceneChromsome==null) {
                 file = getChromosomeFile(sceneId);
            }else {
                 file = getChromosomeFile(sceneId,sceneChromsome.getVersion().toString());
            }


            if (!file.exists()) {
                logger.warn("染色体文件不存在: {}", file.getAbsolutePath());
                return null;
            }

            if (file.length() == 0) {
                logger.warn("染色体文件为空: {}", file.getAbsolutePath());
                return null;
            }

            logger.info("正在从文件加载染色体: {}", file.getAbsolutePath());
            Chromosome chromosome = objectMapper.readValue(file, Chromosome.class);
            logger.info("染色体加载成功，场景ID: {}", sceneId);

            return chromosome;

        } catch (Exception e) {
            // 正确的错误日志写法
            logger.error("加载染色体文件失败，场景ID: " + sceneId, e);
            throw new RuntimeException("加载染色体文件失败: " + e.getMessage(), e);
        }
    }

    /**
     * 将染色体结果保存到JSON文件
     */
    public boolean saveChromosomeToFile(Chromosome chromosome, String sceneId) {
        if (chromosome == null) {
            logger.warn("染色体对象不能为空");
            return false;
        }

        if (sceneId == null || sceneId.trim().isEmpty()) {
            logger.warn("场景ID不能为空");
            return false;
        }

        try {
            ObjectMapper objectMapper = createObjectMapper();

            SceneChromsome sceneChromsome=(SceneChromsome)redisUtils.get("SceneId."+sceneId);
            if(sceneChromsome==null)
            {
                sceneChromsome=new SceneChromsome();
                sceneChromsome.setSceneID(sceneId);
                sceneChromsome.setVersion(1);
            }else {
                Integer v = sceneChromsome.getVersion();
                // 回退后编辑，会清除后续版本，重新生成
                List<SceneDetail> nexts= sceneChromsome.getSceneDetails().stream()
                        .filter(t -> t.getVersion() > v)
                        .collect(Collectors.toList());
                if(nexts!=null&&nexts.size()>0)
                {
                    for (SceneDetail d:nexts) {
                        File file1 = getChromosomeFile(sceneId,d.getVersion().toString());

                        Files.deleteIfExists(file1.toPath());
                    }
                }
                sceneChromsome.getSceneDetails().removeIf(t -> t.getVersion() > v);

                sceneChromsome.setVersion(sceneChromsome.getVersion() + 1);

            }

            SceneDetail sceneDetail=new SceneDetail();
            sceneDetail.setVersion(sceneChromsome.getVersion());
            sceneChromsome.getSceneDetails().add(sceneDetail);


            redisUtils.set("SceneId."+sceneId,sceneChromsome);

            File file = getChromosomeFile(sceneId,sceneChromsome.getVersion().toString());

            File tempFile = new File(file.getParentFile(), file.getName() + ".tmp");

            chromosome.setVersion(sceneChromsome.getVersion());

            objectMapper.writeValue(tempFile, chromosome);

            if (tempFile.length() == 0) {
                logger.error("写入的临时文件为空: {}", tempFile.getAbsolutePath());
                Files.deleteIfExists(tempFile.toPath());
                return false;
            }

            // 验证文件内容
            try {
                Chromosome verifyChromosome = objectMapper.readValue(tempFile, Chromosome.class);
                if (verifyChromosome == null) {
                    throw new IOException("验证读取失败");
                }
            } catch (Exception e) {
                logger.error("验证染色体文件内容失败，文件: {}", tempFile.getAbsolutePath(), e);
                Files.deleteIfExists(tempFile.toPath());
                return false;
            }

            // 替换原文件
            if (file.exists()) {
                Files.deleteIfExists(file.toPath());
            }
            Files.move(tempFile.toPath(), file.toPath());

            logger.info("染色体保存成功，场景ID: {}, 文件: {}", sceneId, file.getAbsolutePath());
            return true;

        } catch (Exception e) {
            logger.error("保存染色体文件失败，场景ID: " + sceneId, e);
            return false;
        }
    }
    public SceneChromsome getSceneChromsome(String sceneId) {
        SceneChromsome sceneChromsome=(SceneChromsome)redisUtils.get("SceneId."+sceneId);

        return sceneChromsome;
    }
    public Chromosome moveChromosome(String sceneId,Integer version)
    {

        SceneChromsome sceneChromsome=(SceneChromsome)redisUtils.get("SceneId."+sceneId);
        if(sceneChromsome==null) {
            return  null;
        }else {

            int prevVersion = sceneChromsome.getVersion() - 1;
            if (prevVersion == 0) {
                throw new RuntimeException("当前已是第一个版本");
            }
            sceneChromsome.setVersion(prevVersion);
            redisUtils.set("SceneId."+sceneId,sceneChromsome);
            Chromosome chromosome = loadChromosomeFromFile(sceneId);
            System.out.println("会退的版本为"+sceneChromsome.getVersion());
            if (chromosome == null) {
                sceneChromsome.setVersion(sceneChromsome.getVersion() + 1);
                redisUtils.set("SceneId."+sceneId,sceneChromsome);
                throw new RuntimeException("回退失败，未找到上一个版本");

            }
            return chromosome;

        }


    }

    /**
     * 检查染色体文件是否存在
     */
    public boolean chromosomeFileExists(String sceneId) {
        if (sceneId == null || sceneId.trim().isEmpty()) {
            return false;
        }
        SceneChromsome sceneChromsome=(SceneChromsome)redisUtils.get("SceneId."+sceneId);
        File file;
        if(sceneChromsome==null) {
            file = getChromosomeFile(sceneId);
        }else {
            file = getChromosomeFile(sceneId,sceneChromsome.getVersion().toString());
        }
        return file.exists() && file.length() > 0;
    }

    /**
     * 删除染色体文件
     */
    public boolean deleteChromosomeFile(String sceneId) {
        try {


            SceneChromsome sceneChromsome=(SceneChromsome)redisUtils.get("SceneId."+sceneId);
            File file;
            if(sceneChromsome==null) {
                file = getChromosomeFile(sceneId);
                if (file.exists()) {
                    boolean deleted = Files.deleteIfExists(file.toPath());
                    if (deleted) {
                        logger.info("删除染色体文件成功: {}", file.getAbsolutePath());
                    }
                    return deleted;
                }
            }else {
                for (SceneDetail d:sceneChromsome.getSceneDetails()) {
                    file = getChromosomeFile(sceneId,d.getVersion().toString());
                    if (file.exists()) {
                        boolean deleted = Files.deleteIfExists(file.toPath());
                        if (deleted) {
                            logger.info("删除染色体文件成功: {}", file.getAbsolutePath());
                        }

                    }
                    return true;
                }

            }


            return true;
        } catch (Exception e) {
            logger.error("删除染色体文件失败，场景ID: " + sceneId, e);
            return false;
        }
    }




}
