package com.aps.service.impl;

import com.aps.common.util.R;
import com.aps.entity.*;
import com.aps.mapper.RoutingDetailMapper;
import com.aps.mapper.RoutingHeaderMapper;
import com.aps.service.*;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 设备日历服务实现类
 * 负责生成设备日历数据、处理订单数据转换等功能
 */
@Slf4j
@Service
@RequiredArgsConstructor  // 使用构造器注入替代@Autowired
public class LanuchServiceImpl implements LanuchService {
    
    @Autowired
    ProdEquipSpecialCalService prodEquipSpecialCalService;
    @Autowired
    ApsOrderService apsOrderService;

    @Autowired
    RoutingHeaderService routingHeaderService;
    @Autowired
    ProdLaunchOrderService prodLaunchOrderService;
    @Autowired
    RoutingDetailMapper routingDetailMapper;
    @Autowired
    ProdProcessExecService prodProcessExecService;

    @Autowired
    RoutingDetailConnectService routingDetailConnectService;
    @Autowired
    ProdOrderProcessService prodOrderProcessService;

    @Autowired
    RoutingHeaderMapper routingHeaderMapper;
    @Autowired
    ProdSceneConfigService prodSceneConfigService;
    /**
     * 生成场景数据
     *
     *
     * @param sceneName        场景名称
     * @param username         操作用户
     * @return 处理结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public R<String> lanuch( String sceneName, String username) {
        try {
            // 参数校验
            if (sceneName == null || sceneName.trim().isEmpty()) {
                return R.failed("场景名称不能为空");
            }
            

            
            // 1. 创建场景
            String sceneId = createScene(sceneName);
            if (sceneId == null) {
                return R.failed("场景名称已存在");
            }

            prodEquipSpecialCalService.copyFromEquipCapacityDef(sceneId);

            List<ApsOrder> apsOrders = apsOrderService.lambdaQuery()
                    .eq(ApsOrder::getIsdeleted, 0)
                    .like(ApsOrder::getCode, "佟礼测试数据")
                    .list();
            if (CollectionUtils.isEmpty(apsOrders)) {
                return R.failed("工单列表不能为空");
            }
            // 4. 处理订单数据转换
            List<ProdLaunchOrder> prodLaunchOrders = processOrderData(sceneId, apsOrders);


            // 5. 转换工单于工序执行表
            convertToProcessExecBatch(prodLaunchOrders, sceneId);

            // 6. 生成工序关联关系
            generateProcessRelationsBatch(prodLaunchOrders, sceneId);
            
            log.info("场景数据生成完成，场景ID：{}", sceneId);
            return R.ok("场景数据生成成功");
        } catch (Exception e) {
            log.error("生成场景数据失败", e);
            return R.failed("生成场景数据失败: " + e.getMessage());
        }
    }
    




    /**
     * 处理订单数据：批量转换ProdOrderMain到ProdLaunchOrder
     *
     * @param sceneId 场景ID
     */
    private List<ProdLaunchOrder> processOrderData(String sceneId, List<ApsOrder> prodOrderMainList) {


        // 批量转换并插入
        List<ProdLaunchOrder> launchOrderList = prodOrderMainList.stream()
                .map(order -> convertToLaunchOrder(order, sceneId))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        // 在您的 LanuchServiceImpl 中修改这一行：
// 原来的问题代码：
// prodLaunchOrderService.saveBatch(launchOrders);

// 修改为：
        if (!launchOrderList .isEmpty()) {
            for (ProdLaunchOrder order : launchOrderList ) {
                try {
                    prodLaunchOrderService.save(order); // 单条插入
                } catch (Exception e) {
                    log.error("保存订单失败: {}", order.getOrderId(), e);
                }
            }
        }


        log.info("成功处理{}条订单数据", prodOrderMainList.size());

        return launchOrderList;
    }

    /**
     * 将ProdOrderMain转换为ProdLaunchOrder
     *
     * @param order    源订单
     * @param sceneId  场景ID
     * @return 转换后的ProdLaunchOrder
     */
    private ProdLaunchOrder convertToLaunchOrder(ApsOrder order, String sceneId) {
        try {
            ProdLaunchOrder launchOrder = new ProdLaunchOrder();
            launchOrder.setOrderId(order.getId());
            launchOrder.setSceneId(sceneId);
            launchOrder.setSerie(order.getSeries());
            launchOrder.setCreateUser(String.valueOf(order.getCreatoruserid()));
            launchOrder.setMaterialCode(order.getMmcode());
//            launchOrder.setStartDate(order.get);
            launchOrder.setEndDate(order.getDeliverytime());
//            launchOrder.setOrderPriority(order.getPrioritry());
            launchOrder.setOrderPriority(1);
            launchOrder.setQuantity(order.getQuantity());
            launchOrder.setMaterialId(order.getMmid());
            
            String mmid = order.getMmid();
            // 通过mmid查找对应的工艺
            if (mmid != null && !mmid.isEmpty()) {
                LambdaQueryWrapper<RoutingHeader> routingWrapper = new LambdaQueryWrapper<>();
                routingWrapper.eq(RoutingHeader::getMaterialId, mmid);
                routingWrapper.eq(RoutingHeader::getIsDeleted, 0); // 添加判断 is_deleted=0
                List<RoutingHeader> routingHeaders = routingHeaderService.list(routingWrapper);
                RoutingHeader routingHeader = null;
                if (!CollectionUtils.isEmpty(routingHeaders)) {
                    routingHeader = routingHeaders.get(0);
                }
                
                if (routingHeader != null) {
                    launchOrder.setRoutingId(routingHeader.getId().toString());
                    launchOrder.setRoutingCode(defaultIfBlank(routingHeader.getCode(), ""));
                }
            }

            return launchOrder;
        } catch (Exception e) {
            log.error("转换订单[{}]失败", order.getId(), e);
            return null;
        }
    }

    /**
     * 创建场景
     *
     * @param sceneName 场景名称
     * @return 场景ID，如果场景已存在则返回null
     */
    private String createScene(String sceneName) {
        // 检查场景名称是否已存在
        boolean exists = prodSceneConfigService.lambdaQuery()
                .eq(ProdSceneConfig::getSceneName, sceneName)
                .exists();

        if (exists) {
            log.warn("场景名称[{}]已存在", sceneName);
            return null;
        }

        // 创建新场景
        String sceneId = generateSceneId();
        ProdSceneConfig sceneConfig = new ProdSceneConfig();
        sceneConfig.setSceneId(sceneId);
        sceneConfig.setSceneName(sceneName);
        sceneConfig.setSceneStatus((short) 1);
        sceneConfig.setCreateTime(LocalDateTime.now());

        prodSceneConfigService.save(sceneConfig);
        log.info("创建新场景成功，场景ID：{}，名称：{}", sceneId, sceneName);

        return sceneId;
    }

    /**
     * 生成场景ID
     *
     * @return 格式化的场景ID
     */
    private String generateSceneId() {
        return UUID.randomUUID().toString().replace("-", "").toUpperCase();
    }

    /**
     * 生成日历数据
     *
     * @param sceneId           场景ID
     * @param planResource      设备信息
     * @param shiftWorkSchedList 班次信息
     * @param startTime         开始时间
     * @param endTime           结束时间
     */
    private void generateCalendarData(String sceneId, PlanResource planResource,
                                     List<MesShiftWorkSched> shiftWorkSchedList,
                                     LocalDateTime startTime, LocalDateTime endTime) {
        LocalDate currentDate = startTime.toLocalDate();
        LocalDate endDate = endTime.toLocalDate();

        // 批量保存列表
        List<ProdEquipSpecialCal> calendarList = new java.util.ArrayList<>();

        while (!currentDate.isAfter(endDate)) {
            int dayOfWeekValue = currentDate.getDayOfWeek().getValue();

            for (MesShiftWorkSched shiftWorkSched : shiftWorkSchedList) {
                if (shiftWorkSched.getStartWeekDay() == dayOfWeekValue) {
                    processSingleShift(sceneId, planResource, shiftWorkSched,
                                     currentDate, startTime, endTime, calendarList);
                }
            }

            currentDate = currentDate.plusDays(1);
        }

        // 批量保存日历数据
        if (!CollectionUtils.isEmpty(calendarList)) {
            prodEquipSpecialCalService.saveBatch(calendarList);
            log.info("设备[{}]生成{}条日历记录", planResource.getReferenceCode(), calendarList.size());
        }
    }

    /**
     * 处理单个班次
     *
     * @param sceneId         场景ID
     * @param planResource    设备信息
     * @param shiftWorkSched  班次信息
     * @param currentDate     当前日期
     * @param startTime       开始时间
     * @param endTime         结束时间
     * @param calendarList    日历列表
     */
    private void processSingleShift(String sceneId, PlanResource planResource,
                                   MesShiftWorkSched shiftWorkSched, LocalDate currentDate,
                                   LocalDateTime startTime, LocalDateTime endTime,
                                   List<ProdEquipSpecialCal> calendarList) {
        LocalTime shiftStartTimeRaw = shiftWorkSched.getShiftStart().toLocalTime();
        LocalTime shiftEndTimeRaw = shiftWorkSched.getShiftEnd().toLocalTime();

        LocalDateTime shiftStartTime = LocalDateTime.of(currentDate, shiftStartTimeRaw);
        LocalDateTime shiftEndTime = LocalDateTime.of(currentDate, shiftEndTimeRaw);

        // 处理跨天班次
        if (shiftEndTimeRaw.isBefore(shiftStartTimeRaw)) {
            shiftEndTime = shiftEndTime.plusDays(1);
        }

        // 时间范围校验
        if (shiftStartTime.isAfter(endTime) || shiftEndTime.isBefore(startTime)) {
            return;
        }

        // 调整时间到范围内
        LocalDateTime actualStartTime = shiftStartTime.isBefore(startTime) ? startTime : shiftStartTime;
        LocalDateTime actualEndTime = shiftEndTime.isAfter(endTime) ? endTime : shiftEndTime;

        if (actualStartTime.isBefore(actualEndTime)) {
            ProdEquipSpecialCal equipSpecialCal = createCalendarRecord(sceneId, planResource,
                                                                      shiftWorkSched, actualStartTime, actualEndTime);
            calendarList.add(equipSpecialCal);
        }
    }

    /**
     * 创建日历记录
     *
     * @param sceneId         场景ID
     * @param planResource    设备信息
     * @param shiftWorkSched  班次信息
     * @param startTime       开始时间
     * @param endTime         结束时间
     * @return 日历记录
     */
    private ProdEquipSpecialCal createCalendarRecord(String sceneId, PlanResource planResource,
                                                    MesShiftWorkSched shiftWorkSched,
                                                    LocalDateTime startTime, LocalDateTime endTime) {
        ProdEquipSpecialCal equipSpecialCal = new ProdEquipSpecialCal();
        equipSpecialCal.setSceneId(sceneId);
        equipSpecialCal.setEquipId(planResource.getReferenceId());
        equipSpecialCal.setEquipCode(planResource.getReferenceCode());
        equipSpecialCal.setStartDate(startTime);
        equipSpecialCal.setEndDate(endTime);
        equipSpecialCal.setShiftWorkSchedCode(shiftWorkSched.getShiftWorkSchedCode());
        equipSpecialCal.setPeriodDesc(shiftWorkSched.getShiftName());

        return equipSpecialCal;
    }

    /**
     * 批量转换工单于工序执行表
     *
     * @param order 工单列表
     * @param sceneId          场景ID
     */
    private void convertToProcessExecBatch(List<ProdLaunchOrder> order, String sceneId) {
        if (CollectionUtils.isEmpty(order)) {
            log.warn("工单列表为空，跳过工序转换");
            return;
        }


        for (ProdLaunchOrder prodOrderMain : order) {
            try {
                convertToProcessExec(prodOrderMain, sceneId);
            } catch (Exception e) {
                log.error("转换工单[{}]到工序执行表失败", prodOrderMain.getOrderId(), e);
                continue;
            }
        }

        log.info("完成{}个工单的工序转换",order.size());
    }

    /**
     * 转换工单到工序执行表
     *
     * @param prodOrderMain  工单ID
     * @param sceneId   场景ID
     * @return 转换结果
     */
    public boolean convertToProcessExec(ProdLaunchOrder prodOrderMain, String sceneId) {


        // 查询对应的工艺
        RoutingHeader routingHeader = getRoutingHeader(prodOrderMain.getRoutingId());

        // 查询该工艺下的所有工序

        List<RoutingDetail> routingDetails = getRoutingDetails(routingHeader.getId());


        // 批量插入新的工序工单记录
        batchInsertProcessExec(prodOrderMain, routingDetails, sceneId);



        return true;
    }

    /**
     * 获取工艺信息
     *
     * @param routingId 工艺ID
     * @return 工艺信息
     */
    private RoutingHeader getRoutingHeader(String routingId) {
        LambdaQueryWrapper<RoutingHeader> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RoutingHeader::getId, routingId)
               .eq(RoutingHeader::getIsDeleted, 0); // 添加 is_deleted=0 过滤条件
        RoutingHeader routingHeader = routingHeaderMapper.selectOne(wrapper);

        if (routingHeader == null) {
            log.error("未找到对应工艺: {}", routingId);
            throw new RuntimeException("未找到对应工艺: " + routingId);
        }

        return routingHeader;
    }

    /**
     * 获取工艺工序列表
     *
     * @param routingHeaderId 工艺头ID
     * @return 工序列表
     */
    private List<RoutingDetail> getRoutingDetails(Integer routingHeaderId) {
        LambdaQueryWrapper<RoutingDetail> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RoutingDetail::getRoutingHeaderId, routingHeaderId)
               .eq(RoutingDetail::getIsDeleted, 0) // 添加 is_deleted=0 过滤条件
               .orderByAsc(RoutingDetail::getTaskSeq);

        List<RoutingDetail> routingDetails = routingDetailMapper.selectList(wrapper);

        if (CollectionUtils.isEmpty(routingDetails)) {
            log.error("工艺下无工序信息: {}", routingHeaderId);
            throw new RuntimeException("工艺下无工序信息: " + routingHeaderId);
        }

        return routingDetails;
    }


    /**
     * 批量插入工序执行记录
     *
     * @param prodOrderMain 工单信息
     * @param routingDetails 工序列表
     * @param sceneId       场景ID
     */
    private void batchInsertProcessExec(ProdLaunchOrder prodOrderMain,
                                       List<RoutingDetail> routingDetails,
                                       String sceneId) {
        List<ProdProcessExec> processExecList = routingDetails.stream()
                .map(detail -> createProcessExec(prodOrderMain, detail, sceneId))
                .collect(Collectors.toList());

        // 批量插入
        if (!CollectionUtils.isEmpty(processExecList)) {
            prodProcessExecService.saveBatch(processExecList); // 假设MyBatis-Plus支持批量插入
        }
    }

    /**
     * 创建工序执行记录
     *
     * @param prodOrderMain 工单信息
     * @param detail        工序详情
     * @param sceneId       场景ID
     * @return 工序执行记录
     */
    private ProdProcessExec createProcessExec(ProdLaunchOrder prodOrderMain,
                                             RoutingDetail detail,
                                             String sceneId) {
        ProdProcessExec prodProcessExec = new ProdProcessExec();
        prodProcessExec.setExecId(UUID.randomUUID().toString().replace("-", ""));
        prodProcessExec.setOrderId(prodOrderMain.getOrderId());
        prodProcessExec.setRoutingDetailId(String.valueOf(detail.getId()));
        prodProcessExec.setTaskSeq(String.valueOf(detail.getTaskSeq()));
        prodProcessExec.setRoutingDetailName(detail.getName());
        prodProcessExec.setMachineId(detail.getEquipTypeId());
//        prodProcessExec.setResourceGroup(String.valueOf(detail.getResourceId()));
        prodProcessExec.setPlanQty(prodOrderMain.getQuantity());
        prodProcessExec.setTargetStartDate(prodOrderMain.getStartDate());
        prodProcessExec.setTargetEndDate(prodOrderMain.getEndDate());
        prodProcessExec.setSceneId(sceneId);
        prodProcessExec.setSetupTime(detail.getSetupTime());
        prodProcessExec.setChangeLineTime(detail.getChangeLineTime());
        prodProcessExec.setConstTime(detail.getConstTime());
        prodProcessExec.setPreprocessingTime(detail.getPreprocessingTime());
        prodProcessExec.setPostprocessingTime(detail.getPostprocessingTime());


        prodProcessExec.setSpeed(
                Optional.ofNullable(detail.getSingleOut())
                        .filter(out -> detail.getRuntime() != null)
                        .filter(out -> detail.getRuntime().compareTo(BigDecimal.ZERO) != 0)
                        .map(out -> out.divide(detail.getRuntime(), 4, RoundingMode.HALF_UP))
                        .orElse(BigDecimal.ZERO)
        );
        System.out.println(prodProcessExec.getSpeed());
        return prodProcessExec;
    }



    /**
     * 批量生成工序关联关系
     *
     * @param prodOrderMains 工单ID列表
     * @param sceneId          场景ID
     */
    private void generateProcessRelationsBatch(List<ProdLaunchOrder> prodOrderMains, String sceneId) {
        if (CollectionUtils.isEmpty(prodOrderMains)) {
            log.warn("工单列表为空，跳过工序关系生成");
            return;
        }

        generateProcessRelations(prodOrderMains, sceneId);

        log.info("完成{}个工单的工序关系生成", prodOrderMains.size());
    }

    /**
     * 生成工序关联关系
     *
     * @param prodOrderMains 工单列表
     * @param sceneId        场景ID
     */
    public void generateProcessRelations(List<ProdLaunchOrder> prodOrderMains, String sceneId) {
        if (CollectionUtils.isEmpty(prodOrderMains)) {
            return;
        }

        List<ProdOrderProcess> processRelations = new java.util.ArrayList<>();

        for (ProdLaunchOrder prodOrderMain : prodOrderMains) {
            try {
                List<ProdOrderProcess> relations = createProcessRelations(prodOrderMain, sceneId);
                processRelations.addAll(relations);
            } catch (Exception e) {
                log.error("生成工单[{}]工序关系失败", prodOrderMain.getOrderId(), e);
                continue;
            }
        }

        // 批量保存
        if (!CollectionUtils.isEmpty(processRelations)) {
            prodOrderProcessService.saveBatch(processRelations);
        }
    }

    /**
     * 创建工单工序关系
     *
     * @param prodOrderMain 工单信息
     * @param sceneId       场景ID
     * @return 工序关系列表
     */
    private List<ProdOrderProcess> createProcessRelations(ProdLaunchOrder prodOrderMain, String sceneId) {
        LambdaQueryWrapper<RoutingDetailConnect> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(RoutingDetailConnect::getRoutingHeaderId, prodOrderMain.getRoutingId())
               .eq(RoutingDetailConnect::getIsdeleted, 0); // 添加 isdeleted=0 过滤条件

        List<RoutingDetailConnect> connections = routingDetailConnectService.list(wrapper);

        return connections.stream()
                .map(connection -> createProcessRelation(prodOrderMain, connection, sceneId))
                .collect(Collectors.toList());
    }

    /**
     * 创建单个工序关系
     *
     * @param prodOrderMain 工单信息
     * @param connection    工序连接
     * @param sceneId       场景ID
     * @return 工序关系
     */
    private ProdOrderProcess createProcessRelation(ProdLaunchOrder prodOrderMain,
                                                  RoutingDetailConnect connection,
                                                  String sceneId) {
        ProdOrderProcess prodOrderProcess = new ProdOrderProcess();
        prodOrderProcess.setSceneId(sceneId);
        prodOrderProcess.setOrderId(prodOrderMain.getOrderId());
        prodOrderProcess.setTaskSeq(connection.getSourceoperation());
        prodOrderProcess.setTargetOrderId(prodOrderMain.getOrderId());
        prodOrderProcess.setTargetTaskSeq(connection.getDestoperation());

        return prodOrderProcess;
    }

    /**
     * 如果值为null或空白，返回默认值
     *
     * @param value        原始值
     * @param defaultValue 默认值
     * @param <T>          类型
     * @return 处理后的值
     */
    private <T> T defaultIfBlank(T value, T defaultValue) {
        if (value == null) {
            return defaultValue;
        }

        if (value instanceof String && ((String) value).trim().isEmpty()) {
            return defaultValue;
        }

        return value;
    }

    /**
     * 如果值为null，返回默认值
     *
     * @param value        原始值
     * @param defaultValue 默认值
     * @param <T>          类型
     * @return 处理后的值
     */
    private <T> T defaultIfNull(T value, T defaultValue) {
        return value != null ? value : defaultValue;
    }
}