package com.aps.service.plan;

import com.aps.common.util.FileHelper;
import com.aps.common.util.JsonFileReader;
import com.aps.controller.gantt.FileUploadController;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Algorithm.GAScheduleResult;
import com.aps.entity.Algorithm.ScheduleParams;
import com.aps.entity.Algorithm.ScheduleResultDetail;
import com.aps.entity.MesHoliday;
import com.aps.entity.basic.ScheduleChromosome;
import com.aps.entity.Schedule.GenVO;
import com.aps.entity.Schedule.MachineVO;
import com.aps.entity.basic.*;
import com.aps.service.Algorithm.GeneticAlgorithm;
import com.aps.service.MesHolidayService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class PlanResultService {
    // 注入FileUploadController
    @Autowired
    private FileUploadController fileUploadController;

    @Autowired
    private MesHolidayService _MesHolidayService;


    private final LocalDateTime baseTime = LocalDateTime.of(2025, 10, 1, 0, 0, 0);

    public List<ScheduleChromosome> execute() {
        try {
            // 1. 读取数据
            List<Machine> machines = loadData("machines.json", Machine.class);
            List<Product> products = loadData("products.json", Product.class);
            List<Order> orders = loadData("orders.json", Order.class);
            
            // 设置机器信息到班次中
            for (Machine machine : machines) {
                if (machine.getShifts() != null) {
                    for (Shift shift : machine.getShifts()) {
                        shift.setMachineId(machine.getId());
                        shift.setMachineName(machine.getName());
                    }
                }
                
                // 调试：打印机器和班次信息
                System.out.println("Machine: " + machine.getId() + ", Name: " + machine.getName());
                if (machine.getShifts() != null) {
                    for (Shift shift : machine.getShifts()) {
                        System.out.println("  Shift: " + shift.getStartTime() + " - " + shift.getEndTime() + 
                                         ", Status: " + shift.getStatus() +
                                         ", MachineId: " + shift.getMachineId() +
                                         ", MachineName: " + shift.getMachineName());
                    }
                }
            }
            
            // 创建节假日
            List<Holiday> holidays = Arrays.asList(
                    new Holiday(LocalDateTime.of(2025, 10, 1, 0, 0),
                            LocalDateTime.of(2025, 10, 7, 23, 59))
            );
            
            // 将节假日添加到所有设备中
            addHolidaysToAllMachines(machines, holidays);

            // 3. 创建调度服务
            MachineSchedulerService machineScheduler = new MachineSchedulerService(
                    holidays, LocalDateTime.of(2025, 10, 1, 0, 0, 0));

            // 4. 初始化机器时间线
            for (Machine machine : machines) {
                MachineTimeline timeline = machineScheduler.getOrCreateTimeline(machine);
                machine.setAvailability(timeline.getSegments());
            }

            // 5. 执行调度算法
            AlgorithmScheduler7 scheduler = new AlgorithmScheduler7(products, machines, orders, machineScheduler);
            List<ScheduleChromosome> scheduleChromosomes = scheduler.RunAll();
            
            // 对调度结果按照 fitness 由高到低排序
            scheduleChromosomes.sort((c1, c2) -> Double.compare(c2.getFitness(), c1.getFitness()));
            
            // 为每个 ScheduleChromosome 分配场景ID（基于排序后的位置）
            for (int i = 0; i < scheduleChromosomes.size(); i++) {
                scheduleChromosomes.get(i).setSceneId(i + 1); // 场景ID从1开始
            }

            return scheduleChromosomes;

        } catch (Exception e) {
            throw new RuntimeException("调度执行失败", e);
        }
    }

    public List<Chromosome>  execute1() {
        try {


            // 1. 读取数据
            List<Machine> machines = loadData("machines.json", Machine.class);
            List<Product> products = loadData("products.json", Product.class);
            List<Order> orders = loadData("orders.json", Order.class);

            // 设置机器信息到班次中
            for (Machine machine : machines) {
                if (machine.getShifts() != null) {
                    for (Shift shift : machine.getShifts()) {
                        shift.setMachineId(machine.getId());
                        shift.setMachineName(machine.getName());
                    }
                }

                // 调试：打印机器和班次信息
                System.out.println("Machine: " + machine.getId() + ", Name: " + machine.getName());
                if (machine.getShifts() != null) {
                    for (Shift shift : machine.getShifts()) {
                        System.out.println("  Shift: " + shift.getStartTime() + " - " + shift.getEndTime() +
                                ", Status: " + shift.getStatus() +
                                ", MachineId: " + shift.getMachineId() +
                                ", MachineName: " + shift.getMachineName());
                    }
                }
            }
            ScheduleParams param = new ScheduleParams();
            param.setBaseTime(LocalDateTime.of(2025, 10, 1, 0, 0, 0));
            param.setPopulationSize(50);
            param.setMaxIterations(100);

         // List<MesHoliday> holidays=  _MesHolidayService.list();

            // 创建节假日
            List<Holiday> holidays = Arrays.asList(
                    new Holiday(LocalDateTime.of(2025, 10, 1, 0, 0),
                            LocalDateTime.of(2025, 10, 7, 23, 59))
            );

            // 将节假日添加到所有设备中
            addHolidaysToAllMachines(machines, holidays);

            // 3. 创建调度服务
            MachineSchedulerService machineScheduler = new MachineSchedulerService(
                    holidays, param.getBaseTime());

            // 4. 初始化机器时间线
            for (Machine machine : machines) {
                MachineTimeline timeline = machineScheduler.getOrCreateTimeline(machine);
                machine.setAvailability(timeline.getSegments());
            }
            // 3. 构建订单-工序数据
            List<Entry> allOperations = new ArrayList<>();
            Random rnd = new Random(); // 注意：此处变量声明但未使用，可根据实际需求保留或移除

            int id = 1;
            for (Order order : orders) {
                // 假设products是一个List<Product>，根据Product的Id查找对应的产品
                Product product = products.stream()
                        .filter(p -> p.getId() == order.getProductId())
                        .findFirst()
                        .orElseThrow(() -> new IllegalArgumentException("未找到对应产品: " + order.getProductId()));

                int sequence = 1;
                for (Operation o : product.getOperations()) { // 假设Product类有getOperations()方法返回工序列表
                    Entry entry = new Entry();
                    entry.setId(id);
                    entry.setGroupId(order.getId());
                    entry.setSequence(sequence);
                    entry.setMachineOptions(o.getMachineOptions()); // 假设Operation类有获取机器选项的方法
                    entry.setPriority(order.getPriority());
                    entry.setQuantity(order.getQuantity());
                   // entry.setMaterialRequirements(o.getMaterialRequirements()); // 假设Operation类有获取物料需求的方法

                    if (sequence != 1) {
                        entry.getPrevEntryIds().add(id - 1); // 假设Entry类有getPrevEntryIds()返回List<Integer>
                    }

                    allOperations.add(entry);

                    sequence++;
                    id++;
                }
            }
            // 5. 执行调度算法
            GeneticAlgorithm scheduler =new GeneticAlgorithm(machines,orders,null,machineScheduler); //new GeneticAlgorithm(products, machines, orders, machineScheduler);
            List<Chromosome> Chromosomes =scheduler.Run(param,allOperations);


            Chromosomes.forEach(this::WriteScheduleSummary);

            return Chromosomes;

        } catch (Exception e) {
            throw new RuntimeException("调度执行失败", e);
        }
    }
    public void WriteScheduleSummary(Chromosome schedule) {
        // 写入日志
        FileHelper.writeLogFile(String.format("\n=== Schedule Summary === %f", schedule.getFitness()));
        FileHelper.writeLogFile(String.format("Operation: %s", schedule.getOperationStr()));
        FileHelper.writeLogFile(String.format("Makespan: %f minutes", schedule.getMakespan()));
        FileHelper.writeLogFile(String.format("Total Tardiness: %f hours", schedule.getDelayTime()));
        FileHelper.writeLogFile(String.format("Setup Time: %f minutes", schedule.getTotalChangeoverTime()));
        FileHelper.writeLogFile(String.format("Flow Time: %f minutes", schedule.getTotalFlowTime()));
        FileHelper.writeLogFile(String.format("Machine Load Balance: %.2f%%", schedule.getMachineLoadStd() * 100));
        FileHelper.writeLogFile("-------------------------");

        // 按订单分组写入
        Map<Integer, List<GAScheduleResult>> orderGroups = schedule.getResult().stream()
                .collect(Collectors.groupingBy(GAScheduleResult::getGroupId));

        for (Map.Entry<Integer, List<GAScheduleResult>> group : orderGroups.entrySet()) {
            List<GAScheduleResult> sortedJobs = group.getValue().stream()
                    .sorted(Comparator.comparingInt(GAScheduleResult::getOperationId))
                    .collect(Collectors.toList());

            for (GAScheduleResult job : sortedJobs) {


                    StringBuilder sb = new StringBuilder();
                    sb.append(String.format(
                            "[%d-%d]:[%s-%s] Order %d,  Machine %d, Operation %d, Batch %d, processingTime %d",
                            job.getStartTime(),
                            job.getEndTime(),
                            ConvertTime(job.getStartTime()),
                            ConvertTime(job.getEndTime()),
                            job.getGroupId(),

                            job.getMachineId(),
                            job.getOperationId(),
                            job.getQuantity(),
                            job.getProcessingTime()
                    ));

                    // 追加基因详情
                    for (ScheduleResultDetail d : job.getGeneDetails()) {
                        sb.append(String.format(
                                "\n\t\t\t\t\t\t\t\t\t\t [%d-%d]:[%s-%s] %d",
                                d.getStartTime(),
                                d.getEndTime(),
                                ConvertTime(d.getStartTime()),
                                ConvertTime(d.getEndTime()),
                                d.getEndTime() - d.getStartTime()
                        ));
                    }

                    FileHelper.writeLogFile(sb.toString());

            }
            FileHelper.writeLogFile("");
        }
    }
    private String ConvertTime(int minute) {
        return baseTime.plusMinutes(minute).format(java.time.format.DateTimeFormatter.ofPattern("MM-dd HH:mm"));
    }

    /**
     * 加载数据，优先从上传文件夹加载，如果不存在则从resources加载
     *
     * @param fileName 文件名
     * @param clazz    类型
     * @param <T>      泛型
     * @return 数据列表
     * @throws IOException IO异常
     */
    private <T> List<T> loadData(String fileName, Class<T> clazz) throws IOException {
        // 检查是否有上传的文件
        if (fileUploadController.isFileUploaded(fileName)) {
            String filePath = fileUploadController.getUploadedFilePath(fileName);
            return JsonFileReader.readListFromFile(filePath, clazz);
        } else {
            // 使用默认的resources文件
            return JsonFileReader.readListFromResources(fileName, clazz);
        }
    }
    
    /**
     * 将假期添加到所有设备的维护窗口中
     * @param machines 设备列表
     * @param holidays 假期列表
     */
    private void addHolidaysToAllMachines(List<Machine> machines, List<Holiday> holidays) {
        if (machines == null || holidays == null) {
            return;
        }
        
        for (Machine machine : machines) {
            // 确保维护窗口列表不为null
            if (machine.getMaintenanceWindows() == null) {
                machine.setMaintenanceWindows(new ArrayList<>());
            }
            
            for (Holiday holiday : holidays) {
                // 将假期转换为维护窗口并添加到设备中
                MaintenanceWindow maintenanceWindow = new MaintenanceWindow(holiday, "Holiday Period");
                machine.addMaintenanceWindow(maintenanceWindow);
            }
        }
    }

    public GenVO convertGeneToGenVO(Gene gene, LocalDateTime baseTime) {
        GenVO genVO = new GenVO();
        genVO.setOrderId(String.valueOf(gene.getOrderId()));
        genVO.setOperationId(gene.getOperationId());
        genVO.setEquipId(gene.getMachineId());
        genVO.setQuantity(BigDecimal.valueOf(gene.getBatchSize()));
        genVO.setStartTime(baseTime.plusMinutes(gene.getStartTime()));
        genVO.setEndTime(baseTime.plusMinutes(gene.getEndTime()));
        genVO.setOperationName(""); // 从其他数据源获取
        genVO.setEquipName(""); // 从其他数据源获取
        return genVO;
    }

    // 批量转换
    public List<GenVO> convertGeneListToGenVO(List<Gene> geneList, LocalDateTime baseTime) {
        return geneList.stream()
                .map(gene -> convertGeneToGenVO(gene, baseTime))
                .collect(Collectors.toList());
    }
    
    public List<MachineVO> convertMachineListToVO(List<Machine> machines) {
        return machines.stream()
                .map(this::convertToVO)
                .collect(Collectors.toList());
    }
    
    private MachineVO convertToVO(Machine machine) {
        MachineVO machineVO = new MachineVO();
        machineVO.setId(machine.getId());
        machineVO.setEquipId(String.valueOf(machine.getId()));
        machineVO.setEquipName(machine.getName());
        // 注意：tasks 字段需要在其他地方设置，因为 Machine 类中没有任务信息
        return machineVO;
    }
}