package com.aps.service.impl;

import com.aps.common.util.SnowFlackIdWorker;
import com.aps.entity.ProdEquipSpecialCal;
import com.aps.entity.EquipCapacityDef;
import com.aps.mapper.ProdEquipSpecialCalMapper;
import com.aps.service.ProdEquipSpecialCalService;
import com.aps.service.EquipCapacityDefService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * <p>
 * 设备特殊日历配置表：记录特定工位/设备在指定时间段的特殊日历规则（如节假日、维护期等） 服务实现类
 * </p>
 *
 * @author MyBatis-Plus
 * @since 2025-11-26
 */
@Service
public class ProdEquipSpecialCalServiceImpl extends ServiceImpl<ProdEquipSpecialCalMapper, ProdEquipSpecialCal> implements ProdEquipSpecialCalService {

    @Autowired
    private EquipCapacityDefService equipCapacityDefService;

    /**
     * 从EquipCapacityDef表中复制is_deleted为0的数据到ProdEquipSpecialCal表
     */
    @Override
    public void copyFromEquipCapacityDef(String sceneId) {
        // 直接通过条件查询，而不是全表查询后过滤
        List<EquipCapacityDef> activeCapacityDefs = equipCapacityDefService.lambdaQuery()
                .eq(EquipCapacityDef::getIsDeleted, 0)
                .isNotNull(EquipCapacityDef::getReferenceId)
                .list();

        // 转换对象并设置必要字段
        List<ProdEquipSpecialCal> specialCals = activeCapacityDefs.stream()
                .map(capacityDef -> convertToProdEquipSpecialCal(capacityDef, sceneId))
                .filter(Objects::nonNull) // 过滤掉转换失败的对象
                .collect(Collectors.toList());

        // 批量保存到ProdEquipSpecialCal表
        if (!specialCals.isEmpty()) {
            // 使用分批插入，避免单次插入过多数据
            int batchSize = 100;
            for (int i = 0; i < specialCals.size(); i += batchSize) {
                int end = Math.min(i + batchSize, specialCals.size());
                this.saveBatch(specialCals.subList(i, end), batchSize);
            }
        }
    }

    /**
     * 将EquipCapacityDef转换为ProdEquipSpecialCal
     * @param capacityDef 源对象
     * @return 转换后的对象
     */
    private ProdEquipSpecialCal convertToProdEquipSpecialCal(EquipCapacityDef capacityDef, String sceneId) {
        try {
            ProdEquipSpecialCal specialCal = new ProdEquipSpecialCal();

            // 设置必填字段，确保不会为空
            specialCal.setId(String.valueOf(SnowFlackIdWorker.getId()));
            specialCal.setSceneId(sceneId); // sceneId不能为空

            // 可选字段的安全转换
            specialCal.setEquipId(capacityDef.getEquipId() != null ? capacityDef.getEquipId().intValue() : null);
            specialCal.setEquipCode(capacityDef.getEquipCode());
            specialCal.setStartDate(capacityDef.getEffectiveStartTime());
            specialCal.setEndDate(capacityDef.getEffectiveEndTime());

            // 处理数值类型转换的空值
            if (capacityDef.getEfficiencyCoeff() != null) {
                specialCal.setEfficiencyCoeff(capacityDef.getEfficiencyCoeff().longValue());
            }

            specialCal.setReferenceId(capacityDef.getReferenceId());
            specialCal.setReferenceName(capacityDef.getReferenceName());
            specialCal.setCreatorUserId(capacityDef.getCreatorUserId());
            specialCal.setCreationTime(LocalDateTime.now());
            specialCal.setReferenceType(capacityDef.getReferenceType());

            return specialCal;
        } catch (Exception e) {
            log.error("转换EquipCapacityDef到ProdEquipSpecialCal失败", e);
            return null;
        }
    }
}