package com.aps.controller.common;

import com.aps.common.util.R;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.basic.Entry;
import com.aps.entity.basic.MachineOption;
import com.aps.entity.common.Paged;
import com.aps.service.common.ChromosomeDataService;
import com.aps.service.plan.PlanResultService;
import com.aps.service.plan.SceneService;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collections;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/queryChromosome")
@Tag(name = "甘特图数据查询", description = "甘特图数据查询")
public class ChromosomeDataController {

    @Autowired
    private ChromosomeDataService chromosomeDataService;
    @Autowired
    private SceneService sceneService;
    @Autowired
    private PlanResultService planResultService;

    /**
     * 通用接口，根据实体名称查询Chromosome中的数据，支持分页和条件过滤
     * 示例:
     * - 文件实体: POST /queryChromosome/order/page
     *   Body: { "sceneId": "SCENE001", "pageIndex": 1, "pageSize": 10, "conditions": [...] }
     * - 数据库实体: POST /queryChromosome/user/page
     *   Body: { "pageIndex": 1, "pageSize": 10, "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine, user, department等)
     * @param paged 分页和条件对象
     * @return 分页数据
     */
    @PostMapping("/{entityName}/page")
    public R<Map<String, Object>> queryChromosomeDataWithConditions(
            @PathVariable String entityName,
            @RequestBody Paged paged) {


            // 获取sceneId（文件实体需要，数据库实体可选）
            String sceneId = paged.getCondition("sceneId") != null ?
                    paged.getCondition("sceneId").getFieldValue() : null;

            // 文件实体必须要有sceneId
            if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
                return R.ok(null);
            }

            Map<String, Object> result = chromosomeDataService.queryChromosomeDataWithConditions(
                    sceneId, entityName, paged);
            return R.ok(result);

    }

    /**
     * 通用接口，根据实体名称查询Chromosome中的列表数据，支持条件过滤
     * 示例:
     * - 文件实体: POST /queryChromosome/order/list
     *   Body: { "sceneId": "SCENE001", "conditions": [...] }
     * - 数据库实体: POST /queryChromosome/user/list
     *   Body: { "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine, user, department等)
     * @param paged 条件对象
     * @return 列表数据
     */
    @PostMapping("/{entityName}/list")
    public R<List<Object>> queryChromosomeDataList(
            @PathVariable String entityName,
            @RequestBody Paged paged) {


        // 获取sceneId（文件实体需要，数据库实体可选）
        String sceneId = paged.getCondition("sceneId") != null ?
                paged.getCondition("sceneId").getFieldValue() : null;

        // 文件实体必须要有sceneId
        if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
            return R.ok(null);
        }

        // 直接调用服务层的list查询方法
        List<Object> result = chromosomeDataService.queryChromosomeDataList(sceneId, entityName, paged);


        if (result == null) {
            return R.ok(Collections.emptyList(), "查询成功，但未找到匹配的数据");
        }

        return R.ok(result);

    }

    /**
     * 通用接口，根据实体名称和ID查询Chromosome中的单个数据
     * 示例:
     * - 文件实体: GET /queryChromosome/order/get?sceneId=xxx&id=123
     * - 数据库实体: GET /queryChromosome/user/get?id=123
     *
     * @param sceneId 场景ID (文件实体必需，数据库实体可选)
     * @param entityName 实体名称
     * @param id 数据ID
     * @return 单个数据对象
     */
    @GetMapping("/{entityName}/get")
    public R<Object> queryChromosomeDataById(
            @RequestParam(required = false) String sceneId,
            @PathVariable String entityName,
            @RequestParam String id) {


            // 文件实体必须要有sceneId
            if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
                return R.failed("文件实体查询时sceneId不能为空");
            }

            Object result = chromosomeDataService.queryChromosomeDataById(sceneId, entityName, id);

            return R.ok(result);



    }

    /**
     * 通用接口，根据实体名称更新Chromosome中的数据
     * 支持文件实体（通过sceneId）和数据库实体的更新
     * 示例:
     * - 文件实体: PUT /queryChromosome/order/update?sceneId=xxx
     *   Body: { "id": 123, "fieldName": "newValue", ... }
     * - 数据库实体: PUT /queryChromosome/user/update
     *   Body: { "id": 123, "fieldName": "newValue", ... }
     *
     * @param sceneId 场景ID (文件实体必需，数据库实体可选)
     * @param entityName 实体名称
     * @param data 要更新的数据 (必须包含id字段)
     * @return 更新结果
     */
    @PostMapping("/{entityName}/update")
    @Operation(summary = "更新实体数据", description = "根据实体名称和ID更新数据，支持文件实体和数据库实体")
    public R<String> updateChromosomeData(
            @Parameter(description = "场景ID (文件实体必需，数据库实体可选)", required = false)
            @RequestParam(required = false) String sceneId,
            @Parameter(description = "taskID (文件实体必需，数据库实体可选)", required = false)
            @RequestParam(required = false) String taskId,
            @Parameter(description = "实体名称", required = true)
            @PathVariable String entityName,
            @Parameter(description = "要更新的数据 (必须包含id字段)", required = true)
            @RequestBody Map<String, Object> data) {

        // 文件实体必须要有sceneId
        if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
           throw new RuntimeException("文件实体更新时sceneId不能为空");
        }
        if (entityName.equalsIgnoreCase("machine")){


            return updateMachineOption(sceneId, taskId, data);
        }


            boolean success = chromosomeDataService.updateChromosomeData(sceneId, entityName, data);
            return R.ok("更新成功");


    }


    private R<String> updateMachineOption(String sceneId, String taskId, Map<String, Object> data) {

            // 1. 加载Chromosome对象
            Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);


            // 2. 根据taskId找到对应的entry
            Entry targetEntry = null;
            for (Entry entry : chromosome.getAllOperations()) {
                if (String.valueOf(entry.getId()).equals(taskId)) {
                    targetEntry = entry;
                    break;
                }
            }

            Long machineId = null;
            Object machineIdObj = data.get("machineId");
            if (machineIdObj instanceof Long) {
                machineId = (Long) machineIdObj;
            } else if (machineIdObj instanceof Integer) {
                machineId = ((Integer) machineIdObj).longValue();
            } else if (machineIdObj instanceof String) {
                machineId = Long.parseLong((String) machineIdObj);
            }

            // 4. 在entry的machineOptions列表中找到对应的machineOption
            MachineOption targetMachineOption = null;
            for (MachineOption machineOption : targetEntry.getMachineOptions()) {
                if (machineOption.getMachineId().equals(machineId)) {
                    targetMachineOption = machineOption;
                    break;
                }
            }



            // 5. 只更新传入的字段，保留原有字段值
            updateMachineOptionFields(targetMachineOption, data);

            // 6. 调用editMachine方法
            planResultService.editMachine(chromosome, sceneId, targetEntry);

            // 7. 保存更新后的Chromosome
            boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);

                return R.ok("更新成功");


    }

    /**
     * 只更新MachineOption的传入字段，保留原有字段值
     */
    private void updateMachineOptionFields(MachineOption machineOption, Map<String, Object> data) {
        // 处理runtime字段
        if (data.containsKey("runtime")) {
            Object runtimeObj = data.get("runtime");
            if (runtimeObj instanceof Integer) {
                machineOption.setRuntime(new BigDecimal((Integer) runtimeObj));
            } else if (runtimeObj instanceof Long) {
                machineOption.setRuntime(new BigDecimal((Long) runtimeObj));
            } else if (runtimeObj instanceof Double) {
                machineOption.setRuntime(new BigDecimal((Double) runtimeObj));
            } else if (runtimeObj instanceof String) {
                machineOption.setRuntime(new BigDecimal((String) runtimeObj));
            }
        }

        // 处理singleOut字段
        if (data.containsKey("singleOut")) {
            Object singleOutObj = data.get("singleOut");
            if (singleOutObj instanceof Integer) {
                machineOption.setSingleOut(new BigDecimal((Integer) singleOutObj));
            } else if (singleOutObj instanceof Long) {
                machineOption.setSingleOut(new BigDecimal((Long) singleOutObj));
            } else if (singleOutObj instanceof Double) {
                machineOption.setSingleOut(new BigDecimal((Double) singleOutObj));
            } else if (singleOutObj instanceof String) {
                machineOption.setSingleOut(new BigDecimal((String) singleOutObj));
            }
        }

        // 处理preTime字段
        if (data.containsKey("preTime")) {
            Object preTimeObj = data.get("preTime");
            if (preTimeObj instanceof Integer) {
                machineOption.setPreTime((Integer) preTimeObj);
            } else if (preTimeObj instanceof Long) {
                machineOption.setPreTime(((Long) preTimeObj).intValue());
            } else if (preTimeObj instanceof String) {
                machineOption.setPreTime(Integer.parseInt((String) preTimeObj));
            }
        }

        // 处理processingTime字段 - 修改为：时间(runtime) ÷ 数量(singleOut)
        if (data.containsKey("processingTime") || data.containsKey("singleOut") || data.containsKey("runtime")) {
            // 1. 获取singleOut值
            BigDecimal singleOut = machineOption.getSingleOut();
            if (data.containsKey("singleOut")) {
                Object singleOutObj = data.get("singleOut");
                if (singleOutObj instanceof Integer) {
                    singleOut = new BigDecimal((Integer) singleOutObj);
                } else if (singleOutObj instanceof Long) {
                    singleOut = new BigDecimal((Long) singleOutObj);
                } else if (singleOutObj instanceof Double) {
                    singleOut = new BigDecimal((Double) singleOutObj);
                } else if (singleOutObj instanceof String) {
                    singleOut = new BigDecimal((String) singleOutObj);
                } else if (singleOutObj instanceof BigDecimal) {
                    singleOut = (BigDecimal) singleOutObj;
                }
            }

            // 2. 获取runtime值
            BigDecimal runtime = machineOption.getRuntime();
            if (data.containsKey("runtime")) {
                Object runtimeObj = data.get("runtime");
                if (runtimeObj instanceof Integer) {
                    runtime = new BigDecimal((Integer) runtimeObj);
                } else if (runtimeObj instanceof Long) {
                    runtime = new BigDecimal((Long) runtimeObj);
                } else if (runtimeObj instanceof Double) {
                    runtime = new BigDecimal((Double) runtimeObj);
                } else if (runtimeObj instanceof String) {
                    runtime = new BigDecimal((String) runtimeObj);
                } else if (runtimeObj instanceof BigDecimal) {
                    runtime = (BigDecimal) runtimeObj;
                }
            }

            // 3. 计算processingTime = runtime ÷ singleOut（时间 ÷ 数量）
            double processingTime = 0.0;
            if (singleOut != null && runtime != null && runtime.compareTo(BigDecimal.ZERO) != 0) {
                // 考虑runtime单位转换：如果runtime是毫秒，转换为秒
                BigDecimal runtimeSeconds = runtime.divide(new BigDecimal(1000), 6, RoundingMode.HALF_UP);
                // 关键修改：时间（秒） ÷ 数量，保留2位小数
                processingTime = runtimeSeconds.divide(singleOut, 2, RoundingMode.HALF_UP).doubleValue();
            }

            // 4. 设置计算后的processingTime
            machineOption.setProcessingTime(processingTime);
        }


//
//        // 处理setupTime字段
//        if (data.containsKey("setupTime")) {
//            Object setupTimeObj = data.get("setupTime");
//            if (setupTimeObj instanceof Integer) {
//                machineOption.setSetupTime((Integer) setupTimeObj);
//            } else if (setupTimeObj instanceof Long) {
//                machineOption.setSetupTime(((Long) setupTimeObj).intValue());
//            } else if (setupTimeObj instanceof String) {
//                machineOption.setSetupTime(Integer.parseInt((String) setupTimeObj));
//            }
//        }
//
//        // 处理teardownTime字段
//        if (data.containsKey("teardownTime")) {
//            Object teardownTimeObj = data.get("teardownTime");
//            if (teardownTimeObj instanceof Integer) {
//                machineOption.setTeardownTime((Integer) teardownTimeObj);
//            } else if (teardownTimeObj instanceof Long) {
//                machineOption.setTeardownTime(((Long) teardownTimeObj).intValue());
//            } else if (teardownTimeObj instanceof String) {
//                machineOption.setTeardownTime(Integer.parseInt((String) teardownTimeObj));
//            }
//        }
//
//        // 处理contantTime字段
//        if (data.containsKey("contantTime")) {
//            Object contantTimeObj = data.get("contantTime");
//            if (contantTimeObj instanceof Integer) {
//                machineOption.setContantTime((Integer) contantTimeObj);
//            } else if (contantTimeObj instanceof Long) {
//                machineOption.setContantTime(((Long) contantTimeObj).intValue());
//            } else if (contantTimeObj instanceof String) {
//                machineOption.setContantTime(Integer.parseInt((String) contantTimeObj));
//            }
//        }

//        // 处理equipCode字段
//        if (data.containsKey("equipCode")) {
//            machineOption.setEquipCode(String.valueOf(data.get("equipCode")));
//        }
//
//        // 处理resourceCode字段
//        if (data.containsKey("resourceCode")) {
//            machineOption.setResourceCode(String.valueOf(data.get("resourceCode")));
//        }

        // 忽略id等未知字段
    }


    private boolean updateMachineOptionsForEntry(Entry targetEntry, List<?> machineOptions, ObjectMapper objectMapper) {
        boolean allSuccess = true;

        for (Object optionObj : machineOptions) {
            if (!(optionObj instanceof Map)) {
                allSuccess = false;
                continue;
            }

            Map<?, ?> optionMap = (Map<?, ?>) optionObj;

            // 获取machineId
            Object machineIdObj = optionMap.get("machineId");
            if (machineIdObj == null) {
                allSuccess = false;
                continue;
            }

            String machineOptionId = String.valueOf(machineIdObj);

            // 在entry的machineOptions列表中找到对应的machineOption
            MachineOption targetMachineOption = null;
            for (MachineOption machineOption : targetEntry.getMachineOptions()) {
                if (String.valueOf(machineOption.getMachineId()).equals(machineOptionId)) {
                    targetMachineOption = machineOption;
                    break;
                }
            }

            if (targetMachineOption == null) {
                allSuccess = false;
                continue;
            }

            // 更新machineOption的字段
            MachineOption updatedMachineOption = objectMapper.convertValue(optionMap, MachineOption.class);

            // 替换原来的machineOption
            int index = targetEntry.getMachineOptions().indexOf(targetMachineOption);
            targetEntry.getMachineOptions().set(index, updatedMachineOption);
        }

        return allSuccess;
    }



    /**
     * 判断是否为文件实体
     */
    private boolean isFileEntity(String entityName) {
        // 这里列出所有文件实体的名称
        String[] fileEntities = {"order", "entry", "machine", "globaloperationinfo", "groupresult", "prodprocessexec", "machineoption"};
        for (String fileEntity : fileEntities) {
            if (fileEntity.equalsIgnoreCase(entityName)) {
                return true;
            }
        }
        return false;
    }
}