package com.aps.controller;

import com.aps.entity.*;
import com.aps.mapper.EquipinfoMapper;
import com.aps.mapper.MpsplannedorderMapper;
import com.aps.mapper.RoutingDetailMapper;
import com.aps.mapper.RoutingHeaderMapper;
import com.aps.service.*;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/schedule")
@Tag(name = "调度任务", description = "调度任务相关接口")
public class ScheduleController {

    @Autowired
    private ScheduleService schedulerService;
    
    @Autowired
    private MpsplannedorderService orderService;
    
    @Autowired
    private RoutingHeaderService processService;
    
    @Autowired
    private RoutingDetailService operationService;
    
    @Autowired
    private RoutingDetailEquipService operationEquipService;
    
    @Autowired
    private EquipinfoService equipmentService;

    @GetMapping("/run")
    public List<ScheduledTask> runScheduling() {
        try {
            // 获取routingid为'SY1102507-0056202车间-21号线'和'SY1102507-0060202车间-21号线'的待调度工单
            List<Mpsplannedorder> orders = orderService.lambdaQuery()
                    .in(Mpsplannedorder::getRoutingid, "SY1102507-0056202车间-21号线", "SY1102507-0060202车间-21号线")
                    .list();
            
            if (orders.isEmpty()) {
                return new ArrayList<>();
            }
            
            // 提取所需的工艺ID列表
            Set<String> routingIds = orders.stream()
                    .map(Mpsplannedorder::getRoutingid)
                    .collect(Collectors.toSet());
            
            // 根据工单获取相关工艺
            List<RoutingHeader> processes = processService.lambdaQuery()
                    .in(RoutingHeader::getCode, routingIds)
                    .list();
            
            if (processes.isEmpty()) {
                throw new RuntimeException("未找到相关工艺信息");
            }
            
            // 提取工艺ID用于后续查询
            Set<Integer> processIds = processes.stream()
                    .map(RoutingHeader::getId)
                    .collect(Collectors.toSet());
            
            // 根据工艺获取相关工序
            List<RoutingDetail> operations = operationService.lambdaQuery()
                    .in(RoutingDetail::getRoutingHeaderId, processIds)
                    .orderByAsc(RoutingDetail::getTaskSeq)
                    .list();
            
            if (operations.isEmpty()) {
                throw new RuntimeException("未找到相关工序信息");
            }
            
            // 提取工序ID用于后续查询
            Set<Long> operationIds = operations.stream()
                    .map(RoutingDetail::getId)
                    .collect(Collectors.toSet());
            
            // 获取相关工序设备关系
            List<RoutingDetailEquip> operationEquipments = operationEquipService.lambdaQuery()
                    .in(RoutingDetailEquip::getRoutingDetailId, operationIds)
                    .list();
            
            if (operationEquipments.isEmpty()) {
                throw new RuntimeException("未找到工序设备关系信息");
            }
            
            // 提取设备ID用于后续查询
            Set<Long> equipmentIds = operationEquipments.stream()
                    .map(RoutingDetailEquip::getEquipId)
                    .collect(Collectors.toSet());
            
            // 获取相关设备
            List<Equipinfo> equipments = equipmentService.lambdaQuery()
                    .in(Equipinfo::getId, equipmentIds)
                    .list();
            
            // 添加空值检查
            if (orders == null || processes == null || operations == null || operationEquipments == null || equipments == null) {
                throw new RuntimeException("获取调度数据失败，存在空数据");
            }

            List<ScheduledTask> result = schedulerService.scheduleOrders(
                    orders, processes, operations, operationEquipments, equipments);
            return result;
        } catch (Exception e) {
            throw new RuntimeException("调度执行失败: " + e.getMessage(), e);
        }
    }
}