package com.aps.service.Algorithm;

import com.aps.entity.Algorithm.IDAndChildID.GroupResult;
import com.aps.entity.Algorithm.IDAndChildID.NodeInfo;
import lombok.Data;

import java.util.*;
import java.util.stream.Collectors;


/**
 * 作者：佟礼
 * 时间：2025-11-29
 */
public class IdGroupingWithDualSerial {
    /**
     * 递归获取ID的所有子孙节点（去重）
     */
    /**
     * 递归获取ID的所有子孙节点（去重）
     */
    private static Set<String> getAllDescendants(String id, Map<String, Set<String>> parentToChildren, Set<String> visited) {
        Set<String> descendants = new HashSet<>();
        if (visited.contains(id) || !parentToChildren.containsKey(id)) {
            return descendants;
        }
        visited.add(id);
        Set<String> directChildren = parentToChildren.get(id);
        for (String child : directChildren) {
            descendants.add(child);
            descendants.addAll(getAllDescendants(child, parentToChildren, visited));
        }
        return descendants;
    }

    /**
     * 分组“无父级且子孙节点有交集”的ID（兼容空ChildID）
     */
    private static List<List<String>> groupNoParentIds(List<String> idList, List<String> childIdList) {
        // 构建父→子映射（过滤空ChildID，去重）
        Map<String, Set<String>> parentToChildren = new HashMap<>();
        for (int i = 0; i < idList.size(); i++) {
            String parentId = idList.get(i);
            String childId = childIdList.get(i);
            // 跳过空ChildID
            if (childId == null || childId.trim().isEmpty()) {
                continue;
            }
            parentToChildren.computeIfAbsent(parentId, k -> new HashSet<>()).add(childId);
        }

        // 构建子→父映射（过滤空ChildID）
        Map<String, Set<String>> childToParents = new HashMap<>();
        for (int i = 0; i < idList.size(); i++) {
            String parentId = idList.get(i);
            String childId = childIdList.get(i);
            if (childId == null || childId.trim().isEmpty()) {
                continue;
            }
            childToParents.computeIfAbsent(childId, k -> new HashSet<>()).add(parentId);
        }

        Set<String> allChildIds = new HashSet<>();
        for (String childId : childIdList) {
            if (childId != null && !childId.trim().isEmpty()) {
                allChildIds.add(childId);
            }
        }

        List<String> noParentIds = parentToChildren.keySet().stream()
                .filter(id -> !allChildIds.contains(id))
                .collect(Collectors.toList());

        // 处理无子女的孤立节点（如23）
        Set<String> allParentIds = new HashSet<>(idList);
        for (String id : allParentIds) {
            if (!parentToChildren.containsKey(id) && !allChildIds.contains(id)) {
                noParentIds.add(id); // 孤立节点作为根节点
            }
        }

        List<List<String>> groups = new ArrayList<>();
        Set<String> processed = new HashSet<>();
        for (String root : noParentIds) {
            if (processed.contains(root)) continue;
            List<String> group = new ArrayList<>();
            group.add(root);
            processed.add(root);
            Set<String> rootDescendants = getAllDescendants(root, parentToChildren, new HashSet<>());
            for (String other : noParentIds) {
                if (processed.contains(other)) continue;
                Set<String> otherDescendants = getAllDescendants(other, parentToChildren, new HashSet<>());
                Set<String> intersection = new HashSet<>(rootDescendants);
                intersection.retainAll(otherDescendants);
                if (!intersection.isEmpty()) {
                    group.add(other);
                    processed.add(other);
                }
            }
            groups.add(group);
        }
        return groups;
    }
    /**
     * 处理单个分组
     */
    private static GroupResult processSingleGroup(List<String> groupRoots,
                                                  Map<String, Set<String>> parentToChildren,
                                                  Map<String, Set<String>> childToParents,
                                                  int[] globalCounter) {
        // 收集分组内所有节点（包含孤立节点）
        Set<String> groupNodes = new HashSet<>();
        Queue<String> queue = new LinkedList<>(groupRoots);
        while (!queue.isEmpty()) {
            String node = queue.poll();
            if (groupNodes.contains(node)) continue;
            groupNodes.add(node);
            if (parentToChildren.containsKey(node)) {
                queue.addAll(parentToChildren.get(node));
            }
        }

        // BFS遍历分配序号（包含孤立节点）
        List<NodeInfo> nodeInfoList = new ArrayList<>();
        Map<String, Integer> originalToGlobalSerial = new HashMap<>();
        Queue<String> bfsQueue = new LinkedList<>(groupRoots);
        Set<String> visited = new HashSet<>(groupRoots);

        int groupCounter = 1;

        // 处理根节点（包括孤立节点）
        for (String root : groupRoots) {
            originalToGlobalSerial.put(root, globalCounter[0]);
            nodeInfoList.add(new NodeInfo(root, globalCounter[0]++, groupCounter++, new ArrayList<>(), new ArrayList<>()));
        }

        // 处理子节点
        while (!bfsQueue.isEmpty()) {
            String parent = bfsQueue.poll();
            if (parentToChildren.containsKey(parent)) {
                for (String child : parentToChildren.get(parent)) {
                    if (!groupNodes.contains(child) || visited.contains(child)) continue;
                    visited.add(child);
                    originalToGlobalSerial.put(child, globalCounter[0]);
                    List<Integer> parentSerials = childToParents.getOrDefault(child, new HashSet<>()).stream()
                            .filter(p -> originalToGlobalSerial.containsKey(p))
                            .map(originalToGlobalSerial::get)
                            .distinct()
                            .collect(Collectors.toList());
                    nodeInfoList.add(new NodeInfo(child, globalCounter[0]++, groupCounter++, parentSerials, new ArrayList<>()));
                    bfsQueue.add(child);
                }
            }
        }

        // 填充子ID列表（过滤空值）
        for (NodeInfo nodeInfo : nodeInfoList) {
            String originalId = nodeInfo.getOriginalId();
            if (parentToChildren.containsKey(originalId)) {
                List<Integer> childSerials = parentToChildren.get(originalId).stream()
                        .filter(c -> originalToGlobalSerial.containsKey(c))
                        .map(originalToGlobalSerial::get)
                        .distinct()
                        .collect(Collectors.toList());
                nodeInfo.setNewChildIds(childSerials);
            }
        }

        return new GroupResult(nodeInfoList, originalToGlobalSerial);
    }

    /**
     * 主方法：处理包含空ChildID的数据
     */
    public static List<GroupResult> groupAndOrderIds(List<String> idList, List<String> childIdList) {
        List<List<String>> groups = groupNoParentIds(idList, childIdList);

        Map<String, Set<String>> parentToChildren = new HashMap<>();
        Map<String, Set<String>> childToParents = new HashMap<>();
        for (int i = 0; i < idList.size(); i++) {
            String parentId = idList.get(i);
            String childId = childIdList.get(i);
            if (childId == null || childId.trim().isEmpty()) {
                continue;
            }
            parentToChildren.computeIfAbsent(parentId, k -> new HashSet<>()).add(childId);
            childToParents.computeIfAbsent(childId, k -> new HashSet<>()).add(parentId);
        }

        int[] globalCounter = {1};
        List<GroupResult> results = new ArrayList<>();
        for (List<String> groupRoots : groups) {
            GroupResult groupResult = processSingleGroup(groupRoots, parentToChildren, childToParents, globalCounter);
            results.add(groupResult);
        }

        return results;
    }
    /**
     * 简化添加新节点：直接指定新父ID、新子ID
     * @param existingResults 已有结果
     * @param targetGroupIndex 目标分组索引（如分组2为1）
     * @param newId 新节点原始ID
     * @param newParentIds 新节点的父ID列表（全局序号）
     * @param newChildIds 新节点的子ID列表（全局序号）
     * @return 更新后的结果
     */
    /**
     * 简化添加新节点：直接指定新父ID、新子ID（修复全局序号重复问题）
     * @param existingResults 已有结果
     * @param targetGroupIndex 目标分组索引（如分组2为1）
     * @param newId 新节点原始ID
     * @param newParentIds 新节点的父ID列表（全局序号）
     * @param newChildIds 新节点的子ID列表（全局序号）
     * @return 更新后的结果
     */
    public static List<GroupResult> addNode(List<GroupResult> existingResults, int targetGroupIndex,
                                            String newId, List<Integer> newParentIds, List<Integer> newChildIds,String insertAfterOriginalId) {
        // 1. 空值安全检查
        if (existingResults == null || existingResults.isEmpty() || targetGroupIndex < 0 || targetGroupIndex >= existingResults.size()) {
            throw new IllegalArgumentException("无效的现有结果或目标分组索引");
        }
        if (newId == null || newId.trim().isEmpty()) {
            throw new IllegalArgumentException("新节点ID不能为空");
        }
        newParentIds = newParentIds != null ? newParentIds : new ArrayList<>();
        newChildIds = newChildIds != null ? newChildIds : new ArrayList<>();

        // 2. 获取目标分组
        GroupResult targetGroup = existingResults.get(targetGroupIndex);
        List<NodeInfo> nodeList = new ArrayList<>(targetGroup.getNodeInfoList());
        Map<String, Integer> serialMap = new HashMap<>(targetGroup.getOriginalToGlobalSerial());

        // 3. 计算所有分组的最大全局序号（确保全局唯一）
        int maxGlobalSerial = existingResults.stream()
                .flatMap(g -> g.getNodeInfoList().stream())
                .mapToInt(NodeInfo::getGlobalSerial)
                .max()
                .orElse(0);
        int newGlobalSerial = maxGlobalSerial + 1;

        // 4. 避免新序号与目标分组内现有序号冲突（双重保障）
        while (serialMap.containsValue(newGlobalSerial)) {
            newGlobalSerial++;
        }


        // 5. 找到插入位置：优先插入到指定节点之后，否则插入到父节点之后
        int insertIndex = 0;
        // 定位要插入到其后的节点（如"20"）
        if (insertAfterOriginalId != null && !insertAfterOriginalId.trim().isEmpty()) {
            for (int i = 0; i < nodeList.size(); i++) {
                NodeInfo node = nodeList.get(i);
                if (insertAfterOriginalId.equals(node.getOriginalId())) {
                    insertIndex = i + 1; // 插入到指定节点之后
                    break;
                }
            }
        }
        // 若未找到指定节点，回退到父节点之后的逻辑
        if (insertIndex == 0) {
            for (int i = 0; i < nodeList.size(); i++) {
                NodeInfo node = nodeList.get(i);
                if (newParentIds.contains(node.getGlobalSerial())) {
                    insertIndex = i + 1;
                }
            }
        }

        // 6. 创建新节点（分组内序号先设为0，后续重新计算）
        NodeInfo newNode = new NodeInfo(newId, newGlobalSerial, 0, newParentIds, newChildIds);
        nodeList.add(insertIndex, newNode);
        serialMap.put(newId, newGlobalSerial);

        // 7. 更新父节点的子ID列表（添加新节点的全局序号）
        for (NodeInfo node : nodeList) {
            if (newParentIds.contains(node.getGlobalSerial())) {
                List<Integer> childIds = new ArrayList<>(node.getNewChildIds());
                if (!childIds.contains(newGlobalSerial)) {
                    childIds.add(newGlobalSerial);
                    node.setNewChildIds(childIds);
                }
            }
        }

        // 8. 更新子节点的父ID列表（添加新节点的全局序号）
        for (NodeInfo node : nodeList) {
            if (newChildIds.contains(node.getGlobalSerial())) {
                List<Integer> parentIds = new ArrayList<>(node.getNewParentIds());
                if (!parentIds.contains(newGlobalSerial)) {
                    parentIds.add(newGlobalSerial);
                    node.setNewParentIds(parentIds);
                }
            }
        }

        // 9. 重新计算分组内序号（按列表顺序）
        for (int i = 0; i < nodeList.size(); i++) {
            nodeList.get(i).setGroupSerial(i + 1);
        }

        // 10. 替换目标分组并返回结果
        List<GroupResult> newResults = new ArrayList<>(existingResults);
        newResults.set(targetGroupIndex, new GroupResult(nodeList, serialMap));
        return newResults;
    }

    /**
     * 通过全局序号删除节点，并重新计算顺序和父子关系
     * @param existingResults 已有分组结果
     * @param deleteGlobalSerial 待删除节点的全局序号
     * @return 删除后的分组结果
     */
    public static List<GroupResult> deleteNodeByGlobalSerial(List<GroupResult> existingResults, int deleteGlobalSerial) {
        // 1. 定位待删除节点及其所在分组
        NodeInfo deleteNode = null;
        int targetGroupIndex = -1;
        GroupResult targetGroup = null;

        for (int i = 0; i < existingResults.size(); i++) {
            GroupResult group = existingResults.get(i);
            Optional<NodeInfo> nodeOpt = group.getNodeInfoList().stream()
                    .filter(node -> deleteGlobalSerial == node.getGlobalSerial())
                    .findFirst();
            if (nodeOpt.isPresent()) {
                deleteNode = nodeOpt.get();
                targetGroupIndex = i;
                targetGroup = group;
                break;
            }
        }

        if (deleteNode == null) {
            System.out.println("待删除节点不存在（全局序号：" + deleteGlobalSerial + "）");
            return existingResults;
        }

        // 2. 获取待删除节点的关键信息
        String deleteOriginalId = deleteNode.getOriginalId();
        List<Integer> deleteParentIds = deleteNode.getNewParentIds();
        List<Integer> deleteChildIds = deleteNode.getNewChildIds();

        // 3. 从分组中移除节点，并清理序号映射
        List<NodeInfo> newNodeList = targetGroup.getNodeInfoList().stream()
                .filter(node -> deleteGlobalSerial != node.getGlobalSerial())
                .collect(Collectors.toList());
        Map<String, Integer> newSerialMap = new HashMap<>(targetGroup.getOriginalToGlobalSerial());
        newSerialMap.remove(deleteOriginalId);

        // 4. 更新父节点的子ID列表（移除待删除节点的全局序号）
        for (NodeInfo node : newNodeList) {
            if (deleteParentIds.contains(node.getGlobalSerial())) {
                List<Integer> newChildIds = node.getNewChildIds().stream()
                        .filter(id -> id != deleteGlobalSerial)
                        .collect(Collectors.toList());
                node.setNewChildIds(newChildIds);
            }
        }

        // 5. 更新子节点的父ID列表（移除待删除节点的全局序号）
        for (NodeInfo node : newNodeList) {
            if (deleteChildIds.contains(node.getGlobalSerial())) {
                List<Integer> newParentIds = node.getNewParentIds().stream()
                        .filter(id -> id != deleteGlobalSerial)
                        .collect(Collectors.toList());
                node.setNewParentIds(newParentIds);
            }
        }

        // 6. 重新计算分组内序号（顺序顺延）
        for (int i = 0; i < newNodeList.size(); i++) {
            newNodeList.get(i).setGroupSerial(i + 1);
        }

        // 7. （可选）重新计算全局序号（保持连续性，如需可开启）
        // resetGlobalSerial(existingResults, targetGroupIndex, newNodeList);

        // 8. 替换目标分组并返回结果
        List<GroupResult> newResults = new ArrayList<>(existingResults);
        newResults.set(targetGroupIndex, new GroupResult(newNodeList, newSerialMap));
        return newResults;
    }
    /**
     * 添加新数据（支持仅孤立节点的场景）
     * @param existingResults 已有分组结果
     * @param newIdList 新数据的ID列表
     * @param newChildIdList 新数据的ChildID列表（允许空值）
     * @return 包含新分组的结果列表
     */
    public static List<GroupResult> addNewDataWithIsolatedGroup(List<GroupResult> existingResults, List<String> newIdList, List<String> newChildIdList) {
        // 空值安全检查
        if (newIdList == null || newIdList.isEmpty()) {
            return existingResults;
        }
        if (newChildIdList == null) {
            newChildIdList = new ArrayList<>();
        }

        // 1. 初始化数据结构
        Set<String> allNewNodes = new HashSet<>(newIdList); // 所有新节点
        Map<String, String> idToChild = new HashMap<>();    // 临时存储ID与ChildID的映射

        // 构建ID→ChildID映射（处理空值）
        for (int i = 0; i < newIdList.size(); i++) {
            String id = newIdList.get(i);
            String childId = i < newChildIdList.size() ? newChildIdList.get(i) : "";
            if (id != null && !id.trim().isEmpty()) {
                idToChild.put(id, (childId == null ? "" : childId.trim()));
                allNewNodes.add(id); // 确保ID被加入
                if (!childId.trim().isEmpty()) {
                    allNewNodes.add(childId.trim()); // 子节点也加入
                }
            }
        }

        // 2. 分离孤立节点和关联节点
        List<String> isolatedNodes = new ArrayList<>();
        List<String> relationNodes = new ArrayList<>();

        for (String node : allNewNodes) {
            String childId = idToChild.getOrDefault(node, "");
            // 孤立节点：无ChildID，且不是任何节点的ChildID
            boolean isIsolated = childId.isEmpty() && !idToChild.values().contains(node);
            if (isIsolated) {
                isolatedNodes.add(node);
            } else {
                relationNodes.add(node);
            }
        }

        // 3. 获取现有结果的最大全局序号
        int maxGlobalSerial = existingResults.stream()
                .flatMap(g -> g.getNodeInfoList().stream())
                .mapToInt(NodeInfo::getGlobalSerial)
                .max()
                .orElse(0);
        int[] globalCounter = {maxGlobalSerial + 1};

        // 4. 处理关联节点（如果有）
        if (!relationNodes.isEmpty()) {
            GroupResult relationGroup = createRelationGroup(idToChild, relationNodes, globalCounter);
            existingResults.add(relationGroup);
        }

        // 5. 处理孤立节点（即使只有孤立节点也创建分组）
        for (String isolatedNode : isolatedNodes) {
            GroupResult isolatedGroup = createIsolatedGroup(isolatedNode, globalCounter);
            existingResults.add(isolatedGroup);
        }

        return existingResults;
    }
    /**
     * 创建关联节点的分组（20→21→22）
     */
    private static GroupResult createRelationGroup(Map<String, String> idToChild, List<String> relationNodes, int[] globalCounter) {
        // 构建父子映射
        Map<String, Set<String>> parentToChildren = new HashMap<>();
        Map<String, Set<String>> childToParents = new HashMap<>();
        for (String parent : idToChild.keySet()) {
            String child = idToChild.get(parent);
            if (!child.isEmpty()) {
                parentToChildren.computeIfAbsent(parent, k -> new HashSet<>()).add(child);
                childToParents.computeIfAbsent(child, k -> new HashSet<>()).add(parent);
            }
        }

        // 识别根节点（无父级的节点）
        List<String> rootNodes = relationNodes.stream()
                .filter(node -> !childToParents.containsKey(node))
                .collect(Collectors.toList());

        List<NodeInfo> nodeList = new ArrayList<>();
        Map<String, Integer> serialMap = new HashMap<>();
        Queue<String> queue = new LinkedList<>(rootNodes);
        Set<String> visited = new HashSet<>(rootNodes);
        int groupCounter = 1;

        // 处理根节点
        for (String root : rootNodes) {
            serialMap.put(root, globalCounter[0]);
            nodeList.add(new NodeInfo(root, globalCounter[0]++, groupCounter++, new ArrayList<>(), new ArrayList<>()));
        }

        // BFS处理子节点
        while (!queue.isEmpty()) {
            String parent = queue.poll();
            if (parentToChildren.containsKey(parent)) {
                for (String child : parentToChildren.get(parent)) {
                    if (visited.contains(child)) continue;
                    visited.add(child);

                    List<Integer> parentSerials = childToParents.get(child).stream()
                            .map(p -> serialMap.get(p))
                            .collect(Collectors.toList());

                    serialMap.put(child, globalCounter[0]);
                    nodeList.add(new NodeInfo(child, globalCounter[0]++, groupCounter++, parentSerials, new ArrayList<>()));
                    queue.add(child);
                }
            }
        }

        // 填充子ID列表
        for (NodeInfo node : nodeList) {
            String originalId = node.getOriginalId();
            if (parentToChildren.containsKey(originalId)) {
                List<Integer> childSerials = parentToChildren.get(originalId).stream()
                        .map(c -> serialMap.get(c))
                        .collect(Collectors.toList());
                node.setNewChildIds(childSerials);
            }
        }

        return new GroupResult(nodeList, serialMap);
    }

    /**
     * 创建孤立节点的分组（如24）
     */
    private static GroupResult createIsolatedGroup(String isolatedNode, int[] globalCounter) {
        NodeInfo node = new NodeInfo(isolatedNode, globalCounter[0], 1, new ArrayList<>(), new ArrayList<>());
        List<NodeInfo> nodeList = Collections.singletonList(node);
        Map<String, Integer> serialMap = Collections.singletonMap(isolatedNode, globalCounter[0]++);
        return new GroupResult(nodeList, serialMap);
    }
    // 测试（新数据）
    public static void Test() {
        // 新表格数据

        List<String> idList = Arrays.asList("1", "2", "3", "3", "5", "10", "3", "6", "7", "9","23");
        List<String> childIdList = Arrays.asList("2", "3", "4", "4", "10", "3", "11", "7", "8", "7","");

        // 分组并生成结果
        List<GroupResult> results = groupAndOrderIds(idList, childIdList);

        // 新增节点信息
        // 直接指定新节点信息：原始ID=12，新父ID=[8,9]，新子ID=[11]
        String newId = "13";
        List<Integer> newParentIds = Arrays.asList(8, 9); // 6和9的全局序号
        List<Integer> newChildIds = Arrays.asList(11);    // 8的全局序号
        int targetGroupIndex = 1; // 分组2的索引

        // 添加新节点
        results = addNode(results, targetGroupIndex, newId, newParentIds, newChildIds,"");


        List<String> newIdList = Arrays.asList("20", "21", "24");
       List<String> newChildIdList = Arrays.asList("21", "22", "");

        // 添加新数据
         results = addNewDataWithIsolatedGroup(results, newIdList, newChildIdList);

        results = deleteNodeByGlobalSerial(results, 13);



        // 输出结果
        for (int i = 0; i < results.size(); i++) {
            GroupResult groupResult = results.get(i);
            List<NodeInfo> nodeInfoList = groupResult.getNodeInfoList();
            System.out.println("分组" + (i + 1) + "顺序：" + nodeInfoList);

            // 输出每个节点的详细信息
            for (NodeInfo nodeInfo : nodeInfoList) {
                System.out.printf("原始ID：%s → 全局序号：%d，分组内序号：%d，新父ID列表：%s，新子ID列表：%s%n",
                        nodeInfo.getOriginalId(),
                        nodeInfo.getGlobalSerial(),
                        nodeInfo.getGroupSerial(),
                        nodeInfo.getNewParentIds().isEmpty() ? "无" : nodeInfo.getNewParentIds(),
                        nodeInfo.getNewChildIds());
            }
            System.out.println("------------------------");
        }
    }
}
